<?php
/**
 * Plugin Name: Hide Posts Toggle System
 * Description: Ermöglicht PMPro-Mitgliedern das Ausblenden bestimmter Beiträge (Shortcode: [hide_posts])
 * Version: 1.6.1
 * Requires PHP: 8.2
 */

if (!defined('ABSPATH')) { exit; }

class Hide_Posts_Toggle_System {
    private const OPTION_NAME   = 'hide_posts_settings';
    private const USER_META_KEY = 'hide_filtered_posts';

    public function __construct() {
        add_action('admin_menu',            [$this, 'add_admin_menu']);
        add_action('admin_init',            [$this, 'register_settings']);
        add_action('pre_get_posts',         [$this, 'modify_query']);
        add_filter('the_posts',             [$this, 'filter_posts_list'], 10, 2);
        add_shortcode('hide_posts',         [$this, 'render_toggle_frontend']);
        add_action('wp_ajax_toggle_hide_posts', [$this, 'ajax_toggle_hide_posts']);
        add_action('wp_enqueue_scripts',    [$this, 'enqueue_frontend_assets']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
    }

    public static function activate(): void {
        $default = [
            'membership_levels'    => [1, 2],
            'tags'                 => [],
            'categories'           => [],
            'posts'                => [],
            'unauthorized_message' => 'Du kannst auf diese Einstellungen nicht zugreifen, da diese nur Mitgliedern mit einer bestimmten Mitgliedschaft angezeigt werden.',
        ];
        $existing = get_option(self::OPTION_NAME, false);
        if ($existing === false) {
            add_option(self::OPTION_NAME, $default, '', 'no');
        } else {
            if (!is_array($existing)) { $existing = []; }
            update_option(self::OPTION_NAME, array_merge($default, $existing), 'no');
        }
    }

    private function get_settings(): array {
        $default = [
            'membership_levels'    => [1, 2],
            'tags'                 => [],
            'categories'           => [],
            'posts'                => [],
            'unauthorized_message' => 'Du kannst auf diese Einstellungen nicht zugreifen, da diese nur Mitgliedern mit einer bestimmten Mitgliedschaft angezeigt werden.',
        ];
        $opt = get_option(self::OPTION_NAME, []);
        if (!is_array($opt)) { $opt = []; }
        $opt['membership_levels']    = isset($opt['membership_levels']) && is_array($opt['membership_levels']) ? array_values(array_filter(array_map('absint', $opt['membership_levels']))) : $default['membership_levels'];
        $opt['tags']                 = isset($opt['tags']) && is_array($opt['tags']) ? array_values(array_filter(array_map('absint', $opt['tags']))) : [];
        $opt['categories']           = isset($opt['categories']) && is_array($opt['categories']) ? array_values(array_filter(array_map('absint', $opt['categories']))) : [];
        $opt['posts']                = isset($opt['posts']) && is_array($opt['posts']) ? array_values(array_filter(array_map('absint', $opt['posts']))) : [];
        $opt['unauthorized_message'] = isset($opt['unauthorized_message']) && is_string($opt['unauthorized_message']) && $opt['unauthorized_message'] !== '' ? $opt['unauthorized_message'] : $default['unauthorized_message'];
        return array_merge($default, $opt);
    }

    private function get_allowed_membership_levels(): array {
        $s = $this->get_settings();
        return !empty($s['membership_levels']) ? $s['membership_levels'] : [1, 2];
    }

    public function add_admin_menu(): void {
        add_options_page(
            'Beiträge Ausblenden Einstellungen',
            'Beiträge Ausblenden',
            'manage_options',
            'hide-posts-settings',
            [$this, 'render_admin_page']
        );
    }

    public function register_settings(): void {
        register_setting('hide_posts_settings_group', self::OPTION_NAME, [
            'sanitize_callback' => [$this, 'sanitize_settings'],
        ]); // Settings API Registrierung mit eigener Sanitization [web:30]

        // Sektion: Membership
        add_settings_section(
            'hide_posts_membership_section',
            'Mitgliedschafts-Level konfigurieren',
            [$this, 'render_membership_section_description'],
            'hide-posts-settings'
        );
        add_settings_field(
            'hide_posts_membership_levels',
            'Erlaubte Membership Level IDs',
            [$this, 'render_membership_levels_field'],
            'hide-posts-settings',
            'hide_posts_membership_section'
        );

        // Sektion: Inhalte
        add_settings_section(
            'hide_posts_main_section',
            'Inhalte zum Ausblenden konfigurieren',
            [$this, 'render_section_description'],
            'hide-posts-settings'
        );
        add_settings_field(
            'hide_posts_tags',
            'Schlagwörter (Tag-IDs)',
            [$this, 'render_tags_field'],
            'hide-posts-settings',
            'hide_posts_main_section'
        );
        add_settings_field(
            'hide_posts_categories',
            'Kategorien (Kategorie-IDs)',
            [$this, 'render_categories_field'],
            'hide-posts-settings',
            'hide_posts_main_section'
        );
        add_settings_field(
            'hide_posts_single',
            'Einzelne Beiträge (Beitrags-IDs)',
            [$this, 'render_posts_field'],
            'hide-posts-settings',
            'hide_posts_main_section'
        );

        // Sektion: Hinweistext
        add_settings_section(
            'hide_posts_notice_section',
            'Hinweistext für Unberechtigte',
            [$this, 'render_notice_section_description'],
            'hide-posts-settings'
        );
        add_settings_field(
            'hide_posts_unauthorized_message',
            'Hinweistext',
            [$this, 'render_unauthorized_message_field'],
            'hide-posts-settings',
            'hide_posts_notice_section'
        );
    }

    public function sanitize_settings($input): array {
        $current = $this->get_settings();
        if (!is_array($input)) { $input = []; }

        $parse_ids = static function ($value): array {
            $value = is_string($value) ? $value : '';
            $value = sanitize_text_field(wp_unslash($value));
            $parts = preg_split('/[\s,;]+/', $value, -1, PREG_SPLIT_NO_EMPTY);
            $ids   = array_map('absint', (array) $parts);
            $ids   = array_filter($ids);
            return array_values(array_unique($ids));
        };

        $out = $current;
        if (array_key_exists('membership_levels', $input)) {
            $out['membership_levels'] = $parse_ids($input['membership_levels']);
            if (empty($out['membership_levels'])) { $out['membership_levels'] = [1, 2]; }
        }
        if (array_key_exists('tags', $input))       { $out['tags']       = $parse_ids($input['tags']); }
        if (array_key_exists('categories', $input)) { $out['categories'] = $parse_ids($input['categories']); }
        if (array_key_exists('posts', $input))      { $out['posts']      = $parse_ids($input['posts']); }

        if (array_key_exists('unauthorized_message', $input)) {
            $msg = is_string($input['unauthorized_message']) ? wp_unslash($input['unauthorized_message']) : '';
            $out['unauthorized_message'] = sanitize_textarea_field($msg);
            if ($out['unauthorized_message'] === '') {
                $out['unauthorized_message'] = 'Du kannst auf diese Einstellungen nicht zugreifen, da diese nur Mitgliedern mit einer bestimmten Mitgliedschaft angezeigt werden.';
            }
        }

        return $out;
    }

    public function render_admin_page(): void {
        if (!current_user_can('manage_options')) { return; }
        if (isset($_GET['settings-updated'])) {
            add_settings_error('hide_posts_messages', 'hide_posts_message', 'Einstellungen gespeichert', 'updated');
        }
        settings_errors('hide_posts_messages');

        ?>
        <div class="wrap hide-posts-settings">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <?php if (!function_exists('pmpro_hasMembershipLevel')): ?>
                <div class="notice notice-warning"><p><strong>Warnung:</strong> Paid Memberships Pro ist nicht installiert oder aktiviert.</p></div>
            <?php endif; ?>

            <form method="post" action="options.php">
                <?php
                settings_fields('hide_posts_settings_group');
                do_settings_sections('hide-posts-settings');
                submit_button('Einstellungen speichern');
                ?>
            </form>

            <?php if (function_exists('pmpro_getAllLevels')): ?>
                <div class="card" style="margin-top:16px;">
                    <h2>Verfügbare Membership Levels</h2>
                    <?php $levels = pmpro_getAllLevels(true); ?>
                    <?php if (!empty($levels)): ?>
                        <table class="wp-list-table widefat fixed striped">
                            <thead><tr><th>ID</th><th>Name</th><th>Beschreibung</th></tr></thead>
                            <tbody>
                            <?php foreach ($levels as $lvl): ?>
                                <tr>
                                    <td><strong><?php echo esc_html($lvl->id); ?></strong></td>
                                    <td><?php echo esc_html($lvl->name); ?></td>
                                    <td><?php echo esc_html($lvl->description ?: '—'); ?></td>
                                </tr>
                            <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p>Keine Membership Levels gefunden.</p>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    // Admin-Beschreibungen
    public function render_membership_section_description(): void { echo '<p>Definieren Sie, welche PMPro Membership Levels Zugriff auf die Ausblende-Funktion haben sollen.</p>'; }
    public function render_section_description(): void { echo '<p>Geben Sie IDs kommagetrennt ein (z.B. 5, 12, 23). Diese Inhalte können von Mitgliedern ausgeblendet werden.</p>'; }
    public function render_notice_section_description(): void { echo '<p>Text, der angezeigt wird, wenn der Shortcode von nicht berechtigten Nutzern aufgerufen wird.</p>'; }

    // Admin-Felder
    public function render_membership_levels_field(): void {
        $s = $this->get_settings();
        $value = implode(', ', $s['membership_levels']);
        echo '<input type="text" name="'.esc_attr(self::OPTION_NAME).'[membership_levels]" value="'.esc_attr($value).'" class="regular-text" placeholder="z.B. 1, 2">';
    }
    public function render_tags_field(): void {
        $s = $this->get_settings();
        $value = implode(', ', $s['tags']);
        echo '<input type="text" name="'.esc_attr(self::OPTION_NAME).'[tags]" value="'.esc_attr($value).'" class="regular-text" placeholder="z.B. 5, 12, 23">';
    }
    public function render_categories_field(): void {
        $s = $this->get_settings();
        $value = implode(', ', $s['categories']);
        echo '<input type="text" name="'.esc_attr(self::OPTION_NAME).'[categories]" value="'.esc_attr($value).'" class="regular-text" placeholder="z.B. 3, 8, 15">';
    }
    public function render_posts_field(): void {
        $s = $this->get_settings();
        $value = implode(', ', $s['posts']);
        echo '<input type="text" name="'.esc_attr(self::OPTION_NAME).'[posts]" value="'.esc_attr($value).'" class="regular-text" placeholder="z.B. 123, 456, 789">';
        echo '<p class="description">IDs werden aus diesem Feld gelesen, nicht aus dem Code.</p>';
    }
    public function render_unauthorized_message_field(): void {
        $s = $this->get_settings();
        echo '<textarea name="'.esc_attr(self::OPTION_NAME).'[unauthorized_message]" rows="3" class="large-text" placeholder="Du kannst auf diese Einstellungen nicht zugreifen, da diese nur Mitgliedern mit einer bestimmten Mitgliedschaft angezeigt werden.">'.esc_textarea($s['unauthorized_message']).'</textarea>';
    }

    // Query-Filter (Frontend-Listen)
    public function modify_query(\WP_Query $query): void {
        if (is_admin()) { return; }
        if ($query->is_singular()) { return; }
        if ($query->is_feed()) { return; }
        if (!function_exists('pmpro_hasMembershipLevel')) { return; }

        $allowed_levels = $this->get_allowed_membership_levels();
        $user_id = get_current_user_id();
        if (!$this->user_has_allowed_level($user_id, $allowed_levels)) { return; }

        if (get_user_meta($user_id, self::USER_META_KEY, true) !== '1') { return; }

        $s = $this->get_settings();

        $pt = $query->get('post_type');
        if ($pt && $pt !== 'post' && !in_array('post', (array) $pt, true)) {
            // Nur klassische Beiträge filtern; bei Bedarf erweitern
            return;
        }

        $tax_query = $query->get('tax_query') ?: [];
        if (!empty($s['tags'])) {
            $tax_query[] = ['taxonomy' => 'post_tag', 'field' => 'term_id', 'terms' => $s['tags'], 'operator' => 'NOT IN'];
        }
        if (!empty($s['categories'])) {
            $tax_query[] = ['taxonomy' => 'category', 'field' => 'term_id', 'terms' => $s['categories'], 'operator' => 'NOT IN'];
        }
        if (!empty($tax_query)) { $query->set('tax_query', $tax_query); }

        if (!empty($s['posts'])) {
            $post__not_in = $query->get('post__not_in') ?: [];
            $post__not_in = array_merge($post__not_in, $s['posts']);
            $query->set('post__not_in', array_values(array_unique($post__not_in)));
        }
    }

    // Fallback: Entfernt blockierte IDs aus beliebigen Post-Listen
    public function filter_posts_list(array $posts, \WP_Query $query): array {
        if (is_admin() || $query->is_singular()) { return $posts; }
        $allowed_levels = $this->get_allowed_membership_levels();
        $user_id = get_current_user_id();
        if (!$this->user_has_allowed_level($user_id, $allowed_levels)) { return $posts; }
        if (get_user_meta($user_id, self::USER_META_KEY, true) !== '1') { return $posts; }

        $s = $this->get_settings();
        if (empty($s['posts'])) { return $posts; }

        $blocked = array_flip($s['posts']);
        $filtered = [];
        foreach ($posts as $p) {
            if (!isset($blocked[$p->ID])) {
                $filtered[] = $p;
            }
        }
        return $filtered;
    }

    // Robuste Level-Prüfung: primär pmpro_hasMembershipLevel(user_id), Fallback über Level-Liste
    private function user_has_allowed_level(int $user_id, array $allowed_levels): bool {
        if (!$user_id || empty($allowed_levels)) { return false; }
        if (function_exists('pmpro_hasMembershipLevel')) {
            if (pmpro_hasMembershipLevel($allowed_levels, $user_id)) {
                return true;
            }
        }
        if (function_exists('pmpro_getMembershipLevelsForUser')) {
            $levels = pmpro_getMembershipLevelsForUser($user_id, true);
            if (is_array($levels)) {
                foreach ($levels as $l) {
                    if (isset($l->id) && in_array((int) $l->id, $allowed_levels, true)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    // Shortcode: Toggle mit sofortiger UI-Rückmeldung (nie leer)
    public function render_toggle_frontend(array $atts = []): string {
        $s = $this->get_settings();
        $user_id = get_current_user_id();
        $eligible = is_user_logged_in() && $this->user_has_allowed_level($user_id, $this->get_allowed_membership_levels());

        ob_start();
        ?>
        <div class="hide-posts-toggle-wrapper">
            <div class="hide-posts-toggle-container">
                <div class="hide-posts-toggle-header">
                    <h3 class="hide-posts-title">Beitragsfilter</h3>
                </div>

                <?php if ($eligible): ?>
                    <?php $is_enabled = get_user_meta($user_id, self::USER_META_KEY, true) === '1'; ?>
                    <div class="hide-posts-toggle-control">
                        <label class="hide-posts-switch<?php echo $is_enabled ? ' is-on' : ''; ?>" role="switch" aria-checked="<?php echo $is_enabled ? 'true' : 'false'; ?>">
                            <input type="checkbox" id="hide-posts-toggle" <?php checked($is_enabled, true); ?> data-nonce="<?php echo esc_attr(wp_create_nonce('hide_posts_toggle')); ?>">
                            <span class="hide-posts-slider" aria-hidden="true"></span>
                        </label>
                        <span class="hide-posts-status"><?php echo $is_enabled ? 'Filter aktiv' : 'Filter inaktiv'; ?></span>
                    </div>
                    <p class="hide-posts-description">Wenn aktiviert, werden definierte Beiträge auf Startseite, in Archiven und in Suchen ausgeblendet.</p>
                    <div class="hide-posts-message" style="display:none;"></div>
                <?php else: ?>
                    <div class="hide-posts-message info" style="display:block;"><?php echo esc_html($s['unauthorized_message']); ?></div>
                <?php endif; ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    public function ajax_toggle_hide_posts(): void {
        check_ajax_referer('hide_posts_toggle', 'nonce');
        if (!is_user_logged_in()) { wp_send_json_error(['message' => 'Nicht angemeldet.']); }
        if (!function_exists('pmpro_hasMembershipLevel') && !function_exists('pmpro_getMembershipLevelsForUser')) {
            wp_send_json_error(['message' => 'PMPro nicht aktiv.']);
        }
        $user_id = get_current_user_id();
        if (!$this->user_has_allowed_level($user_id, $this->get_allowed_membership_levels())) {
            wp_send_json_error(['message' => 'Keine Berechtigung.']);
        }
        $current_value = get_user_meta($user_id, self::USER_META_KEY, true);
        $new_value = ($current_value === '1') ? '0' : '1';
        update_user_meta($user_id, self::USER_META_KEY, $new_value);
        wp_send_json_success([
            'enabled' => $new_value === '1',
            'message' => $new_value === '1' ? 'Filter aktiviert.' : 'Filter deaktiviert.'
        ]);
    }

    public function enqueue_frontend_assets(): void {
        wp_register_style('hide-posts-toggle-style', false, [], '1.6.1');
        wp_enqueue_style('hide-posts-toggle-style');
        wp_add_inline_style('hide-posts-toggle-style', $this->get_frontend_css()); // Inline-CSS mit hoher Spezifität [web:137]

        if (is_user_logged_in()) {
            wp_enqueue_script('jquery');
            $ajax_obj = 'var hidePostsAjax = { ajax_url: "'.esc_js(admin_url('admin-ajax.php')).'", action: "toggle_hide_posts" };';
            wp_add_inline_script('jquery', $ajax_obj."\n".$this->get_frontend_js());
        }
    }

    public function enqueue_admin_assets(string $hook): void {
        if ($hook !== 'settings_page_hide-posts-settings') { return; }
        wp_register_style('hide-posts-admin-style', false, [], '1.6.1');
        wp_enqueue_style('hide-posts-admin-style');
        wp_add_inline_style('hide-posts-admin-style', $this->get_admin_css());
    }

    private function get_frontend_css(): string {
        return <<<CSS
/* 100% Breite, minimalistisch */
.hide-posts-toggle-wrapper { width: 100%; margin: 16px 0; }
.hide-posts-toggle-container { width: 100%; border: 1px solid #e5e7eb; border-radius: 8px; padding: 16px; background: #ffffff; }
.hide-posts-toggle-header { display: flex; align-items: center; justify-content: space-between; margin-bottom: 12px; }
.hide-posts-title { margin: 0; font-size: 18px; font-weight: 600; color: #111827; }
.hide-posts-description { margin: 10px 0 0; font-size: 14px; color: #4b5563; }

/* Toggle-Layout */
.hide-posts-toggle-control { display: flex; align-items: center; gap: 12px; }
.hide-posts-status { font-size: 14px; color: #111827; }

/* Schalter */
.hide-posts-switch { position: relative; display: inline-block; width: 46px; height: 26px; }
.hide-posts-switch input { position: absolute; opacity: 0; width: 46px; height: 26px; margin: 0; cursor: pointer; }
.hide-posts-switch .hide-posts-slider {
  position: absolute; inset: 0; border-radius: 9999px;
  background: #e5e7eb; transition: background-color 0.2s ease, transform 0.2s ease;
}
.hide-posts-switch .hide-posts-slider:before {
  content: ""; position: absolute; height: 22px; width: 22px; left: 2px; top: 2px;
  border-radius: 9999px; background: #ffffff; box-shadow: 0 1px 2px rgba(0,0,0,.08);
  transition: transform 0.2s ease;
}

/* Aktiv-Zustand: per :checked und zusätzlich via .is-on abgesichert */
.hide-posts-switch input:checked + .hide-posts-slider { background: #16a34a !important; }
.hide-posts-switch input:checked + .hide-posts-slider:before { transform: translateX(20px) !important; }
.hide-posts-switch.is-on .hide-posts-slider { background: #16a34a !important; }
.hide-posts-switch.is-on .hide-posts-slider:before { transform: translateX(20px) !important; }

.hide-posts-message {
  margin-top: 10px; font-size: 14px; padding: 10px 12px; border-radius: 6px;
  border: 1px solid #e5e7eb; background: #f9fafb; color: #111827;
}
.hide-posts-message.info { border-color: #dbeafe; background: #eff6ff; color: #1e3a8a; }

@media (prefers-color-scheme: dark) {
  .hide-posts-toggle-container { background: #111827; border-color: #374151; }
  .hide-posts-title, .hide-posts-status { color: #e5e7eb; }
  .hide-posts-description { color: #d1d5db; }
  .hide-posts-switch .hide-posts-slider { background: #374151; }
  .hide-posts-message { background: #1f2937; border-color: #374151; color: #e5e7eb; }
  .hide-posts-message.info { background: #111827; border-color: #1d4ed8; color: #93c5fd; }
}
CSS;
    }

    private function get_frontend_js(): string {
        return <<<JS
jQuery(function($){
  function syncSwitchState($cb){
    var $sw = $cb.closest('.hide-posts-switch');
    var on = $cb.is(':checked');
    $sw.toggleClass('is-on', on).attr('aria-checked', on ? 'true' : 'false');
  }

  var $cbInit = $('#hide-posts-toggle');
  if ($cbInit.length) {
    syncSwitchState($cbInit);
  }

  $(document).on('change', '#hide-posts-toggle', function(){
    var $cb = $(this),
        $wrap = $cb.closest('.hide-posts-toggle-container'),
        $msg = $wrap.find('.hide-posts-message'),
        $status = $wrap.find('.hide-posts-status');

    // Optimistische UI
    syncSwitchState($cb);
    $msg.hide().removeClass('info').text('');

    $.post(hidePostsAjax.ajax_url, {
      action: hidePostsAjax.action,
      nonce: $cb.data('nonce')
    }).done(function(resp){
      if(resp && resp.success){
        $status.text(resp.data.enabled ? 'Filter aktiv' : 'Filter inaktiv');
        setTimeout(function(){ location.reload(); }, 300);
      } else {
        $cb.prop('checked', !$cb.prop('checked'));
        syncSwitchState($cb);
        $msg.text((resp && resp.data && resp.data.message) || 'Ein Fehler ist aufgetreten.').show();
      }
    }).fail(function(){
      $cb.prop('checked', !$cb.prop('checked'));
      syncSwitchState($cb);
      $msg.text('Verbindungsfehler. Bitte versuchen Sie es erneut.').show();
    });
  });
});
JS;
    }

    private function get_admin_css(): string {
        return <<<CSS
.hide-posts-settings .form-table th { width: 260px; }
.hide-posts-settings .description { color: #646970; font-size: 13px; }
CSS;
    }
}

register_activation_hook(__FILE__, [Hide_Posts_Toggle_System::class, 'activate']);
new Hide_Posts_Toggle_System();
