(function(){
    const { createElement: el, useState, useEffect } = wp.element;
    const { PluginDocumentSettingPanel } = wp.editPost;
    const { registerPlugin } = wp.plugins;
    const { TextControl, PanelRow, Button } = wp.components;
    const { __ } = wp.i18n;
    const { useSelect, useDispatch } = wp.data;

    const ReviewPanel = () => {
        const meta = useSelect(select => select('core/editor').getEditedPostAttribute('meta')) || {};
        const { editPost } = useDispatch('core/editor');

        const [criteria, setCriteria] = useState([]);
        const [price, setPrice] = useState(meta._gooloo_reviewbar_price || '');

        useEffect(() => {
            let cData = meta._gooloo_reviewbar_criteria;
            if (cData) {
                try {
                    setCriteria(JSON.parse(cData));
                } catch(e) { setCriteria([]); }
            } else if (window.GOOLOO_RBE && window.GOOLOO_RBE.defaultCriteria) {
                setCriteria(window.GOOLOO_RBE.defaultCriteria);
            }
        }, []);

        useEffect(() => {
            let total = 0;
            if(criteria.length > 0) {
                criteria.forEach(c => total += parseFloat(c.percent || 0));
                let avg = (total / criteria.length).toFixed(2);
                if(avg.endsWith('.00')) avg = avg.slice(0,-3);
                editPost({ meta: { 
                    _gooloo_reviewbar_criteria: JSON.stringify(criteria),
                    _gooloo_reviewbar_value: parseFloat(avg)
                }});
            } else {
                editPost({ meta: { _gooloo_reviewbar_criteria: '[]' }});
            }
        }, [criteria]);

        const updateCriterion = (idx, field, val) => {
            const newC = [...criteria];
            newC[idx][field] = val;
            setCriteria(newC);
        };

        const removeCriterion = (idx) => {
            const newC = [...criteria];
            newC.splice(idx, 1);
            setCriteria(newC);
        };

        const addCriterion = () => {
            setCriteria([...criteria, {label: '', percent: 100}]);
        };

        return el(PluginDocumentSettingPanel, { name:'gooloo-reviewbar-panel', title: 'Review Bar', className:'gooloo-reviewbar-panel' },
            el(PanelRow, {}, 
                el(TextControl, { label:'Price', type:'number', value: price, onChange:(v)=>{ setPrice(v); editPost({meta:{_gooloo_reviewbar_price:v}}); } })
            ),
            el(PanelRow, {}, 
                el(TextControl, { label:'Total (%) - Auto or Manual', type:'number', value: meta._gooloo_reviewbar_value || '', onChange:(v)=>editPost({meta:{_gooloo_reviewbar_value:v}}), help:'Auto-calculated from criteria or enter directly.' })
            ),
            el('div', {}, 
                el('h4', {}, 'Criteria'),
                criteria.map((c, i) => el('div', { key: i, style:{marginBottom:'10px', borderBottom:'1px solid #eee', paddingBottom:'5px'} },
                    el(TextControl, { placeholder:'Label', value: c.label, onChange:(v)=>updateCriterion(i,'label',v) }),
                    el('div', {style:{display:'flex', gap:'5px', alignItems:'flex-end'}},
                        el(TextControl, { type:'number', min:0, max:100, value: c.percent, onChange:(v)=>updateCriterion(i,'percent',v) }),
                        el(Button, { isDestructive:true, variant:'secondary', onClick:()=>removeCriterion(i) }, 'X')
                    )
                )),
                el(Button, { variant:'primary', onClick: addCriterion, style:{marginTop:'10px'} }, '+ Criterion')
            )
        );
    };

    registerPlugin('gooloo-reviewbar-meta-panel', { render: ReviewPanel });
})();
