<?php
/*
Plugin Name: gooloo.de Review Bar Plugin Enhanced
Description: Multi-criteria rating system with average calculation, customizable settings, Classic & Block Editor support, currency selection, and Schema.org structured data.
Version: 4.0.3
Author: gooloo.de
*/

if ( ! defined( 'ABSPATH' ) ) { exit; }

// Constants
if ( ! defined( 'GOOLOO_RBE_VERSION' ) ) {
    define( 'GOOLOO_RBE_VERSION', '4.0.3' );
}
if ( ! defined( 'GOOLOO_RBE_FILE' ) ) {
    define( 'GOOLOO_RBE_FILE', __FILE__ );
}
if ( ! defined( 'GOOLOO_RBE_DIR' ) ) {
    define( 'GOOLOO_RBE_DIR', plugin_dir_path( __FILE__ ) );
}
if ( ! defined( 'GOOLOO_RBE_URL' ) ) {
    define( 'GOOLOO_RBE_URL', plugin_dir_url( __FILE__ ) );
}

// --- Helper: Currency Registry ---
function gooloo_rbe_get_currencies(){
    return array(
            'EUR' => array('label'=>'Euro (€)','symbol'=>'€'),
            'USD' => array('label'=>'US Dollar ($)','symbol'=>'$'),
            'GBP' => array('label'=>'British Pound (£)','symbol'=>'£'),
            'CHF' => array('label'=>'Swiss Franc (CHF)','symbol'=>'CHF'),
            'JPY' => array('label'=>'Japanese Yen (¥)','symbol'=>'¥'),
            'CNY' => array('label'=>'Chinese Yuan (¥)','symbol'=>'¥'),
            'CAD' => array('label'=>'Canadian Dollar (C$)','symbol'=>'C$'),
            'AUD' => array('label'=>'Australian Dollar (A$)','symbol'=>'A$'),
            'NZD' => array('label'=>'New Zealand Dollar (NZ$)','symbol'=>'NZ$'),
            'BRL' => array('label'=>'Brazilian Real (R$)','symbol'=>'R$'),
            'INR' => array('label'=>'Indian Rupee (₹)','symbol'=>'₹'),
            'RUB' => array('label'=>'Russian Ruble (₽)','symbol'=>'₽'),
            'PLN' => array('label'=>'Polish Złoty (zł)','symbol'=>'zł'),
            'CZK' => array('label'=>'Czech Koruna (Kč)','symbol'=>'Kč'),
            'HUF' => array('label'=>'Hungarian Forint (Ft)','symbol'=>'Ft'),
            'NOK' => array('label'=>'Norwegian Krone (kr)','symbol'=>'kr'),
            'SEK' => array('label'=>'Swedish Krona (kr)','symbol'=>'kr'),
            'DKK' => array('label'=>'Danish Krone (kr)','symbol'=>'kr'),
            'RON' => array('label'=>'Romanian Leu (lei)','symbol'=>'lei'),
            'BGN' => array('label'=>'Bulgarian Lev (лв)','symbol'=>'лв'),
            'HRK' => array('label'=>'Croatian Kuna (kn)','symbol'=>'kn'),
            'RSD' => array('label'=>'Serbian Dinar (дин.)','symbol'=>'дин.'),
            'TRY' => array('label'=>'Turkish Lira (₺)','symbol'=>'₺'),
            'ILS' => array('label'=>'Israeli New Shekel (₪)','symbol'=>'₪'),
            'ZAR' => array('label'=>'South African Rand (R)','symbol'=>'R'),
            'MXN' => array('label'=>'Mexican Peso (MX$)','symbol'=>'MX$'),
            'ARS' => array('label'=>'Argentine Peso (AR$)','symbol'=>'AR$'),
            'CLP' => array('label'=>'Chilean Peso (CLP$)','symbol'=>'CLP$'),
            'COP' => array('label'=>'Colombian Peso (COL$)','symbol'=>'COL$'),
            'PEN' => array('label'=>'Peruvian Sol (S/)','symbol'=>'S/'),
            'UYU' => array('label'=>'Uruguayan Peso ($U)','symbol'=>'$U'),
            'BOB' => array('label'=>'Boliviano (Bs.)','symbol'=>'Bs.'),
            'VEF' => array('label'=>'Venezuelan Bolívar (Bs.)','symbol'=>'Bs.'),
            'KRW' => array('label'=>'South Korean Won (₩)','symbol'=>'₩'),
            'HKD' => array('label'=>'Hong Kong Dollar (HK$)','symbol'=>'HK$'),
            'TWD' => array('label'=>'New Taiwan Dollar (NT$)','symbol'=>'NT$'),
            'SGD' => array('label'=>'Singapore Dollar (S$)','symbol'=>'S$'),
            'PHP' => array('label'=>'Philippine Peso (₱)','symbol'=>'₱'),
            'VND' => array('label'=>'Vietnamese Dong (₫)','symbol'=>'₫'),
            'IDR' => array('label'=>'Indonesian Rupiah (Rp)','symbol'=>'Rp'),
            'PKR' => array('label'=>'Pakistani Rupee (₨)','symbol'=>'₨'),
            'BDT' => array('label'=>'Bangladeshi Taka (৳)','symbol'=>'৳'),
            'UAH' => array('label'=>'Ukrainian Hryvnia (₴)','symbol'=>'₴'),
            'GEL' => array('label'=>'Georgian Lari (₾)','symbol'=>'₾'),
            'KZT' => array('label'=>'Kazakhstani Tenge (₸)','symbol'=>'₸'),
            'AED' => array('label'=>'UAE Dirham (د.إ)','symbol'=>'د.إ'),
            'SAR' => array('label'=>'Saudi Riyal (ر.س)','symbol'=>'ر.س'),
            'QAR' => array('label'=>'Qatari Riyal (ر.ق)','symbol'=>'ر.ق'),
            'OMR' => array('label'=>'Omani Rial (ر.ع.)','symbol'=>'ر.ع.'),
            'KWD' => array('label'=>'Kuwaiti Dinar (د.ك)','symbol'=>'د.ك'),
            'BHD' => array('label'=>'Bahraini Dinar (ب.د)','symbol'=>'ب.د'),
            'EGP' => array('label'=>'Egyptian Pound (E£)','symbol'=>'E£'),
            'MAD' => array('label'=>'Moroccan Dirham (د.م.)','symbol'=>'د.م.'),
            'TND' => array('label'=>'Tunisian Dinar (د.ت)','symbol'=>'د.ت'),
            'DZD' => array('label'=>'Algerian Dinar (د.ج)','symbol'=>'د.ج'),
            'KES' => array('label'=>'Kenyan Shilling (KSh)','symbol'=>'KSh'),
            'NGN' => array('label'=>'Nigerian Naira (₦)','symbol'=>'₦'),
            'GHS' => array('label'=>'Ghanaian Cedi (₵)','symbol'=>'₵'),
            'ETB' => array('label'=>'Ethiopian Birr (Br)','symbol'=>'Br'),
            'TZS' => array('label'=>'Tanzanian Shilling (TSh)','symbol'=>'TSh'),
            'UGX' => array('label'=>'Ugandan Shilling (USh)','symbol'=>'USh'),
            'XOF' => array('label'=>'West African CFA franc (CFA)','symbol'=>'CFA'),
            'XAF' => array('label'=>'Central African CFA franc (FCFA)','symbol'=>'FCFA'),
            'XPF' => array('label'=>'CFP Franc (₣)','symbol'=>'₣'),
            'ISK' => array('label'=>'Icelandic Króna (kr)','symbol'=>'kr'),
            'ALL' => array('label'=>'Albanian Lek (L)','symbol'=>'L'),
            'MKD' => array('label'=>'Macedonian Denar (ден)','symbol'=>'ден'),
            'BYN' => array('label'=>'Belarusian Ruble (Br)','symbol'=>'Br'),
            'AZN' => array('label'=>'Azerbaijani Manat (₼)','symbol'=>'₼'),
            'AMD' => array('label'=>'Armenian Dram (֏)','symbol'=>'֏'),
            'MDL' => array('label'=>'Moldovan Leu (L)','symbol'=>'L'),
            'BAM' => array('label'=>'Bosnia and Herzegovina Convertible Mark (KM)','symbol'=>'KM'),
            'BBD' => array('label'=>'Barbadian Dollar (Bds$)','symbol'=>'Bds$'),
            'JMD' => array('label'=>'Jamaican Dollar (J$)','symbol'=>'J$'),
            'TTD' => array('label'=>'Trinidad and Tobago Dollar (TT$)','symbol'=>'TT$'),
            'DOP' => array('label'=>'Dominican Peso (RD$)','symbol'=>'RD$'),
            'CRC' => array('label'=>'Costa Rican Colón (₡)','symbol'=>'₡'),
            'GTQ' => array('label'=>'Guatemalan Quetzal (Q)','symbol'=>'Q'),
            'HNL' => array('label'=>'Honduran Lempira (L)','symbol'=>'L'),
            'NIO' => array('label'=>'Nicaraguan Córdoba (C$)','symbol'=>'C$'),
            'PAB' => array('label'=>'Panamanian Balboa (B/.)','symbol'=>'B/.'),
            'BZD' => array('label'=>'Belize Dollar (BZ$)','symbol'=>'BZ$'),
            'RWF' => array('label'=>'Rwandan Franc (FRw)','symbol'=>'FRw'),
            'CDF' => array('label'=>'Congolese Franc (FC)','symbol'=>'FC'),
            'ZMW' => array('label'=>'Zambian Kwacha (ZK)','symbol'=>'ZK'),
            'MZN' => array('label'=>'Mozambican Metical (MT)','symbol'=>'MT'),
            'BWP' => array('label'=>'Botswana Pula (P)','symbol'=>'P'),
            'MUR' => array('label'=>'Mauritian Rupee (₨)','symbol'=>'₨'),
            'LKR' => array('label'=>'Sri Lankan Rupee (Rs)','symbol'=>'Rs'),
            'NPR' => array('label'=>'Nepalese Rupee (Rs)','symbol'=>'Rs'),
            'MMK' => array('label'=>'Burmese Kyat (K)','symbol'=>'K'),
            'LAK' => array('label'=>'Lao Kip (₭)','symbol'=>'₭'),
            'KHR' => array('label'=>'Cambodian Riel (៛)','symbol'=>'៛'),
            'MYR' => array('label'=>'Malaysian Ringgit (RM)','symbol'=>'RM'),
            'THB' => array('label'=>'Thai Baht (฿)','symbol'=>'฿'),
            'BND' => array('label'=>'Brunei Dollar (B$)','symbol'=>'B$'),
            'PGK' => array('label'=>'Papua New Guinean Kina (K)','symbol'=>'K'),
            'FJD' => array('label'=>'Fijian Dollar (FJ$)','symbol'=>'FJ$'),
            'WST' => array('label'=>'Samoan Tala (T)','symbol'=>'T'),
            'TOP' => array('label'=>'Tongan Paʻanga (T$)','symbol'=>'T$')
        );
}

function gooloo_rbe_is_valid_currency($code){
    $code = strtoupper(substr( (string) $code, 0, 10 ));
    $list = gooloo_rbe_get_currencies();
    return isset($list[$code]) ? $code : '';
}

function gooloo_rbe_currency_symbol($code){
    $list = gooloo_rbe_get_currencies();
    if(isset($list[$code])) return $list[$code]['symbol'];
    return $code ? $code : 'EUR';
}

// --- Activation ---
register_activation_hook( __FILE__, 'review_bar_plugin_activate' );
function review_bar_plugin_activate() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'review_bar_data';
    $charset_collate = $wpdb->get_charset_collate();
    $sql = "CREATE TABLE $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        post_id bigint(20) NOT NULL,
        value float,
        price float,
        brand varchar(255),
        sku varchar(255),
        PRIMARY KEY  (id),
        UNIQUE KEY post_id (post_id)
    ) $charset_collate;";
    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta( $sql );

    if ( ! get_option( 'review_bar_settings' ) ) {
        $default_options = array(
            'line_color'      => '#ff0080',
            'price_text'      => 'Preis¹: ',
            'rating_text'     => 'Gesamtbewertung: ',
            'additional_html' => '<b>Du erhälst Informationen über diese Anzeige, indem Du <a href="/colorbanner">hier klickst</a>.</b>',
            'divider_style'   => 'border-top: 2px solid {color}; margin: 10px 0;',
            'euro_text'       => ' Euro',
            'default_currency'=> 'EUR',
            'default_criteria'=> '[]'
        );
        add_option( 'review_bar_settings', $default_options );
    }

    update_option( 'review_bar_version', '4.0.3' );
}

// --- Register Meta ---
add_action( 'init', 'gooloo_rbe_register_meta' );
function gooloo_rbe_register_meta() {
    $auth_cb = function() { return current_user_can( 'edit_posts' ); };

    register_post_meta( 'post', '_gooloo_reviewbar_value', array( 'type'=>'number', 'single'=>true, 'show_in_rest'=>true, 'sanitize_callback'=>function($v){ return is_numeric($v)?(float)$v:null; }, 'auth_callback'=>$auth_cb ) );
    register_post_meta( 'post', '_gooloo_reviewbar_price', array( 'type'=>'number', 'single'=>true, 'show_in_rest'=>true, 'sanitize_callback'=>function($v){ return is_numeric($v)?(float)$v:null; }, 'auth_callback'=>$auth_cb ) );
    register_post_meta( 'post', '_gooloo_reviewbar_brand', array( 'type'=>'string', 'single'=>true, 'show_in_rest'=>true, 'sanitize_callback'=>'sanitize_text_field', 'auth_callback'=>$auth_cb ) );
    register_post_meta( 'post', '_gooloo_reviewbar_sku', array( 'type'=>'string', 'single'=>true, 'show_in_rest'=>true, 'sanitize_callback'=>'sanitize_text_field', 'auth_callback'=>$auth_cb ) );
    register_post_meta( 'post', '_gooloo_reviewbar_currency', array( 'type'=>'string', 'single'=>true, 'show_in_rest'=>true, 'sanitize_callback'=>'sanitize_text_field', 'auth_callback'=>$auth_cb ) );

    register_post_meta( 'post', '_gooloo_reviewbar_criteria', array(
        'type' => 'string',
        'single' => true,
        'show_in_rest' => true,
        'sanitize_callback' => function( $v ) {
            $data = json_decode($v, true);
            if(!is_array($data)) return '[]';
            $clean = array();
            foreach($data as $item){
                if(isset($item['label']) && isset($item['percent'])){
                    $clean[] = array(
                        'label' => sanitize_text_field($item['label']),
                        'percent' => max(0, min(100, floatval($item['percent'])))
                    );
                }
            }
            return json_encode($clean, JSON_UNESCAPED_UNICODE);
        },
        'auth_callback' => $auth_cb,
    ) );
}

// --- Settings Page (Admin) ---
add_action('admin_menu', 'review_bar_add_admin_menu');
function review_bar_add_admin_menu() {
    add_options_page( 'Review Bar Settings', 'Review Bar', 'manage_options', 'review-bar-settings', 'review_bar_settings_page' );
}

add_action('admin_init', 'review_bar_settings_init');
function review_bar_settings_init() {
    register_setting('review_bar_settings_group', 'review_bar_settings', 'review_bar_sanitize_settings');
    add_settings_section('review_bar_appearance_section', __('Visual Settings','review-bar'), function(){}, 'review-bar-settings');
    add_settings_section('review_bar_text_section', __('Text & Defaults','review-bar'), function(){}, 'review-bar-settings');

    add_settings_field('line_color', 'Line Color', 'review_bar_line_color_callback', 'review-bar-settings', 'review_bar_appearance_section');
    add_settings_field('price_text', 'Price Text', 'review_bar_price_text_callback', 'review-bar-settings', 'review_bar_text_section');
    add_settings_field('rating_text', 'Rating Text', 'review_bar_rating_text_callback', 'review-bar-settings', 'review_bar_text_section');
    add_settings_field('euro_text', 'Currency Text (legacy)', 'review_bar_euro_text_callback', 'review-bar-settings', 'review_bar_text_section');
    add_settings_field('additional_html', 'Additional HTML', 'review_bar_additional_html_callback', 'review-bar-settings', 'review_bar_text_section');
    add_settings_field('divider_style', 'Divider CSS', 'review_bar_divider_style_callback', 'review-bar-settings', 'review_bar_appearance_section');
    add_settings_field('default_currency', 'Default Currency', 'review_bar_default_currency_callback', 'review-bar-settings', 'review_bar_text_section');
    add_settings_field('default_criteria', 'Standard Criteria', 'review_bar_default_criteria_callback', 'review-bar-settings', 'review_bar_text_section');
}

function review_bar_line_color_callback() { $o=get_option('review_bar_settings'); echo '<input type="text" name="review_bar_settings[line_color]" value="'.esc_attr($o['line_color']??'#ff0080').'" class="review-bar-color-picker" data-default-color="#ff0080">'; }
function review_bar_price_text_callback() { $o=get_option('review_bar_settings'); echo '<input type="text" name="review_bar_settings[price_text]" value="'.esc_attr($o['price_text']??'Preis¹: ').'" class="regular-text">'; }
function review_bar_rating_text_callback() { $o=get_option('review_bar_settings'); echo '<input type="text" name="review_bar_settings[rating_text]" value="'.esc_attr($o['rating_text']??'Gesamtbewertung: ').'" class="regular-text">'; }
function review_bar_euro_text_callback() { $o=get_option('review_bar_settings'); echo '<input type="text" name="review_bar_settings[euro_text]" value="'.esc_attr($o['euro_text']??'' ).'" class="regular-text">'; }
function review_bar_divider_style_callback() { $o=get_option('review_bar_settings'); echo '<textarea name="review_bar_settings[divider_style]" rows="3" class="large-text">'.esc_textarea($o['divider_style']??'').  '</textarea>'; }
function review_bar_additional_html_callback() { $o=get_option('review_bar_settings'); echo '<textarea name="review_bar_settings[additional_html]" rows="3" class="large-text">'.esc_textarea($o['additional_html']??'').  '</textarea>'; }
function review_bar_default_currency_callback(){ 
    $o=get_option('review_bar_settings'); $cur=$o['default_currency']??'EUR'; $list=gooloo_rbe_get_currencies(); 
    echo '<select name="review_bar_settings[default_currency]">'; 
    foreach($list as $c=>$i){ echo '<option value="'.esc_attr($c).'" '.selected($cur,$c,false).'>'.esc_html($i['label']).'</option>'; } 
    echo '</select>'; 
}

function review_bar_default_criteria_callback() {
    $o = get_option('review_bar_settings');
    $criteria_json = isset($o['default_criteria']) ? $o['default_criteria'] : '[]';
    ?>
    <div id="rb-default-criteria-wrapper">
        <input type="hidden" name="review_bar_settings[default_criteria]" id="rb-default-criteria-input" value="<?php echo esc_attr($criteria_json); ?>">
        <div id="rb-criteria-list"></div>
        <button type="button" class="button" id="rb-add-criteria">+ Add Criterion</button>
        <p class="description">These criteria will be auto-loaded in new posts.</p>
    </div>
    <script>
    jQuery(document).ready(function($){
        let criteria = <?php echo $criteria_json ?: '[]'; ?>;
        const list = $('#rb-criteria-list');
        const input = $('#rb-default-criteria-input');

        function render() {
            list.empty();
            criteria.forEach((c, i) => {
                list.append(`
                    <div style="margin-bottom: 5px; display:flex; gap:5px;">
                        <input type="text" placeholder="Label" value="${c.label}" class="rb-crit-label" data-idx="${i}">
                        <button type="button" class="button rb-remove-crit" data-idx="${i}">x</button>
                    </div>
                `);
            });
            input.val(JSON.stringify(criteria));
        }

        $('#rb-add-criteria').on('click', function(){
            criteria.push({label: '', percent: 100});
            render();
        });

        list.on('input', '.rb-crit-label', function(){
            let idx = $(this).data('idx');
            criteria[idx].label = $(this).val();
            input.val(JSON.stringify(criteria));
        });

        list.on('click', '.rb-remove-crit', function(){
            let idx = $(this).data('idx');
            criteria.splice(idx, 1);
            render();
        });

        render();
    });
    </script>
    <?php
}

function review_bar_sanitize_settings($input) {
    $input['line_color'] = sanitize_hex_color($input['line_color']??'#ff0080');
    $input['price_text'] = wp_kses($input['price_text']??'', array());
    $input['rating_text'] = wp_kses($input['rating_text']??'', array());
    $input['euro_text'] = wp_kses($input['euro_text']??'', array());
    $input['additional_html'] = wp_kses_post($input['additional_html']??'');
    $input['divider_style'] = sanitize_textarea_field($input['divider_style']??'');

    if(isset($input['default_currency'])) {
        $cur = strtoupper( preg_replace('/[^A-Z]/','', $input['default_currency'] ) );
        $input['default_currency'] = gooloo_rbe_is_valid_currency( $cur ) ?: 'EUR';
    }

    if(isset($input['default_criteria'])) {
        $data = json_decode($input['default_criteria'], true);
        if(is_array($data)) {
            $clean = [];
            foreach($data as $d) {
                if(!empty($d['label'])) $clean[] = ['label'=>sanitize_text_field($d['label']), 'percent'=>100];
            }
            $input['default_criteria'] = json_encode($clean, JSON_UNESCAPED_UNICODE);
        } else {
            $input['default_criteria'] = '[]';
        }
    }
    return $input;
}

function review_bar_settings_page() { 
    echo '<div class="wrap"><h1>'.esc_html(get_admin_page_title()).'</h1><form action="options.php" method="post">'; 
    settings_fields('review_bar_settings_group'); 
    do_settings_sections('review-bar-settings'); 
    submit_button(); 
    echo '</form></div>'; 
}

add_action('admin_enqueue_scripts', 'review_bar_enqueue_color_picker');
function review_bar_enqueue_color_picker($hook) {
    if ( 'settings_page_review-bar-settings' !== $hook ) return;
    wp_enqueue_style('wp-color-picker');
    wp_enqueue_script('wp-color-picker');
    wp_add_inline_script('wp-color-picker', 'jQuery(function($){ $(".review-bar-color-picker").wpColorPicker(); });');
}

// --- Classic Editor Meta Boxes ---
add_action('add_meta_boxes', 'add_reviewbar_meta_boxes');
function add_reviewbar_meta_boxes() {
    add_meta_box('reviewbar_meta_box', 'Review Bar & Criteria', 'reviewbar_meta_box_callback', 'post', 'advanced', 'high');
    add_meta_box('brand_meta_box', 'Brand', 'brand_meta_box_callback', 'post', 'advanced', 'high');
    add_meta_box('sku_meta_box', 'SKU/EAN/ISBN', 'sku_meta_box_callback', 'post', 'advanced', 'high');
}

function reviewbar_meta_box_callback($post) {
    wp_nonce_field('reviewbar_meta_box_nonce', 'reviewbar_meta_box_nonce');

    $criteria_json = get_post_meta($post->ID, '_gooloo_reviewbar_criteria', true);
    if ( empty($criteria_json) || $criteria_json == '[]' ) {
        $opts = get_option('review_bar_settings');
        $criteria_json = isset($opts['default_criteria']) ? $opts['default_criteria'] : '[]';
    }

    $value = get_post_meta($post->ID, '_gooloo_reviewbar_value', true);
    $price = get_post_meta($post->ID, '_gooloo_reviewbar_price', true);

    $opts = get_option('review_bar_settings');
    $def_cur = $opts['default_currency'] ?? 'EUR';
    $cur = get_post_meta($post->ID, '_gooloo_reviewbar_currency', true) ?: $def_cur;
    $currencies = gooloo_rbe_get_currencies();

    ?>
    <table class="form-table">
        <tr>
            <th scope="row">Price & Currency</th>
            <td>
                <input type="number" step="0.01" name="reviewbar_price" value="<?php echo esc_attr($price); ?>" placeholder="0.00">
                <select name="reviewbar_currency">
                    <?php foreach($currencies as $c => $i): ?>
                        <option value="<?php echo esc_attr($c); ?>" <?php selected($cur, $c); ?>><?php echo esc_html($i['label']); ?></option>
                    <?php endforeach; ?>
                </select>
            </td>
        </tr>
        <tr>
            <th scope="row">Total Rating (%)</th>
            <td>
                <input type="number" id="rb-total-value" name="reviewbar_value" value="<?php echo esc_attr($value); ?>" readonly style="background:#eee;"> 
                <span class="description">Auto-calculated from criteria.</span>
            </td>
        </tr>
        <tr>
            <th scope="row">Criteria</th>
            <td>
                <div id="rb-post-criteria-container"></div>
                <input type="hidden" name="reviewbar_criteria" id="rb-post-criteria-input" value="<?php echo esc_attr($criteria_json); ?>">
                <button type="button" class="button" id="rb-post-add-crit" style="margin-top:10px;">+ Add Criterion</button>
            </td>
        </tr>
    </table>

    <script>
    jQuery(document).ready(function($){
        let criteria = <?php echo $criteria_json ?: '[]'; ?>;
        const container = $('#rb-post-criteria-container');
        const input = $('#rb-post-criteria-input');
        const totalInput = $('#rb-total-value');

        function calculateTotal() {
            if(criteria.length === 0) return 0;
            let sum = 0;
            criteria.forEach(c => sum += parseFloat(c.percent || 0));
            let avg = (sum / criteria.length).toFixed(2);
            if(avg.endsWith('.00')) avg = avg.slice(0, -3);
            totalInput.val(avg);
        }

        function render() {
            container.empty();
            criteria.forEach((c, i) => {
                container.append(`
                    <div class="rb-crit-row" style="margin-bottom:5px; display:flex; gap:10px; align-items:center;">
                        <input type="text" class="rb-c-label" value="${c.label}" data-idx="${i}" placeholder="Name" style="flex:1;">
                        <input type="number" class="rb-c-percent" value="${c.percent}" data-idx="${i}" min="0" max="100" step="1" style="width:70px;"> %
                        <button type="button" class="button rb-c-remove" data-idx="${i}">x</button>
                    </div>
                `);
            });
            input.val(JSON.stringify(criteria));
            calculateTotal();
        }

        $('#rb-post-add-crit').click(function(){
            criteria.push({label: '', percent: 100});
            render();
        });

        container.on('input', '.rb-c-label', function(){ criteria[$(this).data('idx')].label = $(this).val(); input.val(JSON.stringify(criteria)); });
        container.on('input', '.rb-c-percent', function(){ 
            let v = parseFloat($(this).val());
            if(v > 100) v = 100; if(v < 0) v = 0;
            criteria[$(this).data('idx')].percent = v; 
            input.val(JSON.stringify(criteria)); 
            calculateTotal();
        });
        container.on('click', '.rb-c-remove', function(){ criteria.splice($(this).data('idx'), 1); render(); });

        render();
    });
    </script>
    <?php
}

function brand_meta_box_callback($post) {
    global $wpdb; $table = $wpdb->prefix . 'review_bar_data';
    $data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table WHERE post_id = %d", $post->ID ) );
    $brand = $data ? $data->brand : '';
    wp_nonce_field('brand_meta_box_nonce', 'brand_meta_box_nonce');
    echo '<table class="form-table"><tr><th scope="row"><label for="brand">Brand:</label></th><td><input type="text" id="brand" name="brand" value="' . esc_attr($brand) . '" class="regular-text" /></td></tr></table>';
}

function sku_meta_box_callback($post) {
    global $wpdb; $table = $wpdb->prefix . 'review_bar_data';
    $data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table WHERE post_id = %d", $post->ID ) );
    $sku = $data ? $data->sku : '';
    echo '<table class="form-table"><tr><th scope="row"><label for="sku">SKU/EAN/ISBN:</label></th><td><input type="text" id="sku" name="sku" value="' . esc_attr($sku) . '" class="regular-text" /></td></tr></table>';
}

// --- Save Handler (Classic) ---
add_action('save_post', 'save_reviewbar_meta_box_v4', 20, 2);
function save_reviewbar_meta_box_v4($post_id, $post = null) {
    if(!isset($_POST['reviewbar_meta_box_nonce']) || !wp_verify_nonce($_POST['reviewbar_meta_box_nonce'], 'reviewbar_meta_box_nonce')) return;
    if(defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if(!current_user_can('edit_post', $post_id)) return;

    // Criteria save
    $criteria_raw = isset($_POST['reviewbar_criteria']) ? wp_unslash($_POST['reviewbar_criteria']) : '[]';
    $criteria_arr = json_decode($criteria_raw, true);
    if(!is_array($criteria_arr)) $criteria_arr = [];

    $clean_crit = [];
    $total = 0;
    $count = 0;

    foreach($criteria_arr as $c) {
        if(!empty($c['label'])) {
            $p = floatval($c['percent']);
            $clean_crit[] = ['label'=>sanitize_text_field($c['label']), 'percent'=>$p];
            $total += $p;
            $count++;
        }
    }

    $avg = $count > 0 ? round($total / $count, 2) : 0;
    $value = $avg;

    $price = isset($_POST['reviewbar_price']) ? floatval(str_replace(',', '.', $_POST['reviewbar_price'])) : 0;
    $brand = isset($_POST['brand']) ? trim($_POST['brand']) : '';
    $sku = isset($_POST['sku']) ? sanitize_text_field($_POST['sku']) : '';
    $currency = isset($_POST['reviewbar_currency']) ? sanitize_text_field($_POST['reviewbar_currency']) : 'EUR';

    update_post_meta($post_id, '_gooloo_reviewbar_criteria', json_encode($clean_crit, JSON_UNESCAPED_UNICODE));
    update_post_meta($post_id, '_gooloo_reviewbar_value', $value);
    update_post_meta($post_id, '_gooloo_reviewbar_price', $price);
    update_post_meta($post_id, '_gooloo_reviewbar_brand', $brand);
    update_post_meta($post_id, '_gooloo_reviewbar_sku', $sku);
    update_post_meta($post_id, '_gooloo_reviewbar_currency', $currency);

    update_post_meta($post_id, 'reviewbar_value', $value);
    update_post_meta($post_id, 'reviewbar_price', $price);
    update_post_meta($post_id, 'brand', $brand);
    update_post_meta($post_id, 'sku', $sku);

    global $wpdb; $table = $wpdb->prefix . 'review_bar_data';
    $exists = $wpdb->get_var($wpdb->prepare("SELECT id FROM $table WHERE post_id=%d", $post_id));

    $data = array(
        'post_id' => $post_id,
        'value' => $value,
        'price' => $price,
        'brand' => $brand,
        'sku' => $sku
    );

    if($exists) $wpdb->update($table, $data, array('post_id'=>$post_id), array('%d','%f','%f','%s','%s'), array('%d'));
    else $wpdb->insert($table, $data, array('%d','%f','%f','%s','%s'));
}

// --- REST API Sync ---
add_action( 'rest_after_insert_post', 'gooloo_rbe_rest_calc_sync', 10, 3 );
function gooloo_rbe_rest_calc_sync( $post, $request, $creating ) {
    if('post' !== $post->post_type) return;

    $post_id = $post->ID;
    $crit_json = get_post_meta($post_id, '_gooloo_reviewbar_criteria', true);
    $data = json_decode($crit_json, true);

    if(is_array($data) && count($data) > 0) {
        $sum = 0; $cnt = 0;
        foreach($data as $d) { $sum += floatval($d['percent']); $cnt++; }
        $avg = round($sum / $cnt, 2);

        update_post_meta($post_id, '_gooloo_reviewbar_value', $avg);

        global $wpdb; $table = $wpdb->prefix.'review_bar_data';
        $price = get_post_meta($post_id, '_gooloo_reviewbar_price', true);
        $curr = get_post_meta($post_id, '_gooloo_reviewbar_currency', true);

        $exists = $wpdb->get_var($wpdb->prepare("SELECT id FROM $table WHERE post_id=%d", $post_id));
        $db_data = ['value'=>$avg, 'price'=>$price, 'post_id'=>$post_id];
        if($exists) $wpdb->update($table, $db_data, ['post_id'=>$post_id]);
        else $wpdb->insert($table, $db_data);
    }
}

// --- Helper Get Data from DB ---
function gooloo_rbe_get_data( $post_id ) {
    global $wpdb; $table = $wpdb->prefix . 'review_bar_data';
    $row = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table WHERE post_id = %d", $post_id ) );
    if ( ! $row ) return array( 'value'=>0, 'price'=>0, 'brand'=>'', 'sku'=>'' );
    return array( 'value'=>$row->value, 'price'=>$row->price, 'brand'=>$row->brand, 'sku'=>$row->sku );
}

// --- Frontend Output ---
function reviewbar_shortcode( $atts ) {
    $atts = shortcode_atts( array( 'value'=>'', 'price'=>'', 'post_id'=>'', 'currency'=>'' ), $atts, 'reviewbar' );

    $pid = $atts['post_id'] !== '' ? (int) $atts['post_id'] : ( is_singular('post') ? get_the_ID() : 0 );
    if(!$pid) return '';

    $criteria = [];
    $d = gooloo_rbe_get_data($pid);

    if($atts['value'] === '') $atts['value'] = $d['value'];
    if($atts['price'] === '') $atts['price'] = $d['price'];
    if($atts['currency'] === '') $atts['currency'] = get_post_meta($pid, '_gooloo_reviewbar_currency', true);

    $c_json = get_post_meta($pid, '_gooloo_reviewbar_criteria', true);
    if($c_json) $criteria = json_decode($c_json, true);

    if ( empty($atts['value']) || ((float)$atts['value'] == 0 && empty($criteria)) ) return '';

    $opts = get_option('review_bar_settings');
    $line_color = $opts['line_color'] ?? '#ff0080';

    $val = floatval($atts['value']);
    $color = ($val<=30)?'red':(($val<=60)?'orange':(($val<=75)?'yellow':'green'));

    $cur = $atts['currency'] ?: ($opts['default_currency']??'EUR');
    $sym = gooloo_rbe_currency_symbol($cur);
    $price_fmt = number_format((float)$atts['price'], 2, ',', '.') . ' ' . $sym;

    $out = '<div class="gooloo-review-wrapper">';

    $out .= '<div class="gooloo-reviewbar-container main-bar">';
    $out .= '<div class="gooloo-reviewbar-divider" style="border-top: 2px solid '.esc_attr($line_color).'; margin: 10px 0;"></div>';
    $out .= '<div class="rb-header"><span class="rb-price">'.esc_html(($opts['price_text']??'Preis: ') . $price_fmt).'</span> | <span class="rb-rating">'.esc_html(($opts['rating_text']??'Rating: ') . $val . '%').'</span></div>';
    $out .= '<div class="gooloo-reviewbar" style="background-color:'.esc_attr($color).'; width:'.esc_attr($val).'%; height:20px; border-radius:10px;"></div>';
    $out .= '</div>';

    if( !empty($criteria) && is_array($criteria) ) {
        $out .= '<div class="rb-criteria-grid">';
        foreach($criteria as $c) {
            $c_val = floatval($c['percent']);
            $c_col = ($c_val<=30)?'red':(($c_val<=60)?'orange':(($c_val<=75)?'yellow':'green'));

            $out .= '<div class="rb-criteria-item">';
            $out .= '<div class="rb-c-info"><span class="rb-c-label">'.esc_html($c['label']).'</span> <span class="rb-c-val">'.esc_html($c_val).'%</span></div>';
            $out .= '<div class="rb-c-bar-bg" style="background:#eee; height:8px; border-radius:4px; overflow:hidden;">';
            $out .= '<div class="rb-c-bar-fill" style="background:'.esc_attr($c_col).'; width:'.esc_attr($c_val).'%; height:100%;"></div>';
            $out .= '</div>';
            $out .= '</div>';
        }
        $out .= '</div>';
    }

    if(!empty($opts['additional_html'])) $out .= '<div class="rb-add-html">'.$opts['additional_html'].'</div>';
    $out .= '</div>';

    return $out;
}
add_shortcode( 'reviewbar', 'reviewbar_shortcode' );

// --- Auto-append to content ---
add_filter('the_content', 'add_reviewbar_shortcode');
function add_reviewbar_shortcode($content) {
    if ( ! is_singular('post') ) return $content;
    global $wpdb, $post; 
    $table = $wpdb->prefix . 'review_bar_data';
    $data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table WHERE post_id = %d", $post->ID ) );

    if ( $data && ( $data->value !== null || get_post_meta($post->ID, '_gooloo_reviewbar_criteria', true) ) ) {
        $cur = get_post_meta( $post->ID, '_gooloo_reviewbar_currency', true );
        $content .= do_shortcode( '[reviewbar value="' . esc_attr($data->value??0) . '" price="' . esc_attr($data->price??0) . '" currency="' . esc_attr($cur) . '"]' );
    }
    return $content;
}

// --- Schema.org JSON-LD Output (CRITICAL FIX v4.0.3) ---
add_action('wp_head', 'gooloo_rbe_schema_output', 99);
function gooloo_rbe_schema_output() {
    if ( ! is_singular('post') ) return;

    global $post, $wpdb;
    $table = $wpdb->prefix . 'review_bar_data';
    $data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table WHERE post_id = %d", $post->ID ) );

    if ( ! $data ) return;

    $value = (float)($data->value ?? 0);
    $price = (float)($data->price ?? 0);
    $brand = sanitize_text_field($data->brand ?? '');
    $sku = sanitize_text_field($data->sku ?? '');
    $currency = get_post_meta($post->ID, '_gooloo_reviewbar_currency', true);
    $currency = $currency ?: get_option('review_bar_settings')['default_currency'] ?? 'EUR';

    // Build schema
    $schema = array(
        '@context' => 'https://schema.org/',
        '@type' => 'Product',
        'name' => get_the_title($post->ID),
        'description' => get_the_excerpt($post->ID),
        'url' => get_permalink($post->ID),
        'review' => array(
            '@type' => 'Review',
            'author' => array(
                '@type' => 'Organization',
                'name' => get_bloginfo('name')
            ),
            'datePublished' => get_the_date('Y-m-d', $post->ID),
            'reviewRating' => array(
                '@type' => 'Rating',
                'ratingValue' => $value,
                'bestRating' => 100,
                'worstRating' => 0
            )
        ),
        'aggregateRating' => array(
            '@type' => 'AggregateRating',
            'ratingValue' => $value,
            'bestRating' => 100,
            'worstRating' => 0,
            'ratingCount' => 1
        ),
        'offers' => array(
            '@type' => 'Offer',
            'url' => get_permalink($post->ID),
            'price' => $price,
            'priceCurrency' => $currency,
            'availability' => 'https://schema.org/InStock'
        )
    );

    // Add image if available
    if (has_post_thumbnail($post->ID)) {
        $thumb_id = get_post_thumbnail_id($post->ID);
        $img_url = wp_get_attachment_image_url($thumb_id, 'full');
        if ($img_url) {
            $schema['image'] = $img_url;
        }
    }

    // Add brand if available
    if (!empty($brand)) {
        $schema['brand'] = array(
            '@type' => 'Brand',
            'name' => $brand
        );
    }

    // Add SKU if available
    if (!empty($sku)) {
        $schema['sku'] = $sku;
    }

    echo '<script type="application/ld+json">' . json_encode($schema, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) . '</script>' . "\n";
}

// --- Frontend CSS ---
add_action('wp_head', 'gooloo_rbe_frontend_css');
function gooloo_rbe_frontend_css() {
    $options = get_option('review_bar_settings');
    $line_color = isset($options['line_color']) ? $options['line_color'] : '#ff0080';
    echo '<style>
        .gooloo-review-wrapper { width: 100%; margin-bottom: 20px; }
        .gooloo-reviewbar-container { width: 100%; padding: 10px; box-sizing: border-box; }
        .gooloo-reviewbar-divider { border-top: 2px solid ' . esc_attr($line_color) . '; margin: 10px 0; }
        .rb-header { font-weight: bold; font-size: 1.1em; margin-bottom: 5px; }
        .rb-criteria-grid { display: flex; flex-wrap: wrap; gap: 20px; margin-top: 15px; }
        .rb-criteria-item { flex: 1 1 45%; min-width: 200px; }
        .rb-c-info { display: flex; justify-content: space-between; font-size: 0.9em; margin-bottom: 2px; }
        .rb-criteria-item:last-child:nth-child(odd) { flex-basis: 100%; }
    </style>';
}

// --- Settings Link ---
add_filter( 'plugin_action_links_' . plugin_basename(__FILE__), 'review_bar_add_settings_link' );
function review_bar_add_settings_link( $links ) { 
    $settings_link = '<a href="' . esc_url( admin_url('options-general.php?page=review-bar-settings') ) . '">Settings</a>'; 
    array_unshift( $links, $settings_link ); 
    return $links; 
}

// --- Version Check ---
add_action('admin_init', 'review_bar_check_version');
function review_bar_check_version() { 
    $installed = get_option('review_bar_version','1.0'); 
    if ( version_compare( $installed, GOOLOO_RBE_VERSION, '<' ) ) { 
        update_option('review_bar_version', GOOLOO_RBE_VERSION); 
    } 
}

// --- Editor Assets ---
add_action( 'enqueue_block_editor_assets', 'gooloo_rbe_enqueue_editor_assets' );
function gooloo_rbe_enqueue_editor_assets() {
    wp_enqueue_script( 'gooloo-reviewbar-editor', GOOLOO_RBE_URL . 'assets/js/editor.js', array( 'wp-element','wp-components','wp-data','wp-edit-post','wp-plugins','wp-i18n' ), GOOLOO_RBE_VERSION, true );
    $opts = get_option('review_bar_settings');
    $def_crit = isset($opts['default_criteria']) ? $opts['default_criteria'] : '[]';
    wp_localize_script( 'gooloo-reviewbar-editor', 'GOOLOO_RBE', array( 
        'defaultCriteria' => json_decode($def_crit, true),
        'defaultCurrency' => $opts['default_currency'] ?? 'EUR'
    ));
}
?>
