<?php
/*
Plugin Name: gooloo.de Review Bar Plugin Enhanced
Description: Fetches and stores review bar data in a custom database table with customizable settings, Classic & Block Editor support, and Schema.org structured data.
Version: 3.0
Author: gooloo.de
*/

if ( ! defined( 'ABSPATH' ) ) { exit; }

// Constants
if ( ! defined( 'GOOLOO_RBE_VERSION' ) ) {
    define( 'GOOLOO_RBE_VERSION', '3.0' );
}
if ( ! defined( 'GOOLOO_RBE_FILE' ) ) {
    define( 'GOOLOO_RBE_FILE', __FILE__ );
}
if ( ! defined( 'GOOLOO_RBE_DIR' ) ) {
    define( 'GOOLOO_RBE_DIR', plugin_dir_path( __FILE__ ) );
}
if ( ! defined( 'GOOLOO_RBE_URL' ) ) {
    define( 'GOOLOO_RBE_URL', plugin_dir_url( __FILE__ ) );
}

// Activation: create table, migrate, defaults, version
register_activation_hook( __FILE__, 'review_bar_plugin_activate' );
function review_bar_plugin_activate() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'review_bar_data';
    $charset_collate = $wpdb->get_charset_collate();
    $sql = "CREATE TABLE $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        post_id bigint(20) NOT NULL,
        value float,
        price float,
        brand varchar(255),
        sku varchar(255),
        PRIMARY KEY  (id),
        UNIQUE KEY post_id (post_id)
    ) $charset_collate;";
    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta( $sql );
    transfer_existing_data();
    if ( ! get_option( 'review_bar_settings' ) ) {
        $default_options = array(
            'line_color'      => '#ff0080',
            'price_text'      => 'Preis¹: ',
            'rating_text'     => 'Gesamtbewertung: ',
            'additional_html' => '<b>Du erhälst Informationen über diese Anzeige, indem Du <a href="/colorbanner">hier klickst</a>.</b>',
            'divider_style'   => 'border-top: 2px solid {color}; margin: 10px 0;',
            'euro_text'       => ' Euro'
        );
        add_option( 'review_bar_settings', $default_options );
    }
    update_option( 'review_bar_version', '3.0' );
}

function transfer_existing_data() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'review_bar_data';
    $posts = get_posts( array( 'post_type' => 'post', 'numberposts' => -1, 'fields' => 'ids' ) );
    foreach ( $posts as $post_id ) {
        $value = get_post_meta( $post_id, 'reviewbar_value', true );
        $price = get_post_meta( $post_id, 'reviewbar_price', true );
        $brand = get_post_meta( $post_id, 'brand', true );
        $sku   = get_post_meta( $post_id, 'sku', true );
        if ( $value !== '' || $price !== '' || $brand !== '' || $sku !== '' ) {
            $wpdb->replace( $table_name, array(
                'post_id' => $post_id,
                'value'   => $value !== '' ? (float) $value : null,
                'price'   => $price !== '' ? (float) $price : null,
                'brand'   => $brand,
                'sku'     => $sku,
            ) );
        }
    }
}

// Register post meta for Block Editor compatibility and REST access
add_action( 'init', 'gooloo_rbe_register_meta' );
function gooloo_rbe_register_meta() {
    $auth_cb = function() { return current_user_can( 'edit_posts' ); };
    register_post_meta( 'post', '_gooloo_reviewbar_value', array(
        'type' => 'number', 'single' => true, 'show_in_rest' => true,
        'sanitize_callback' => function( $v ) { $v = is_numeric( $v ) ? (float) $v : null; if ( $v === null ) return null; return max( 0, min( 100, $v ) ); },
        'auth_callback' => $auth_cb,
    ) );
    register_post_meta( 'post', '_gooloo_reviewbar_price', array(
        'type' => 'number', 'single' => true, 'show_in_rest' => true,
        'sanitize_callback' => function( $v ) { $v = is_numeric( $v ) ? (float) $v : null; if ( $v === null ) return null; return max( 0, $v ); },
        'auth_callback' => $auth_cb,
    ) );
    register_post_meta( 'post', '_gooloo_reviewbar_brand', array(
        'type' => 'string', 'single' => true, 'show_in_rest' => true,
        'sanitize_callback' => 'sanitize_text_field',
        'auth_callback' => $auth_cb,
    ) );
    register_post_meta( 'post', '_gooloo_reviewbar_sku', array(
        'type' => 'string', 'single' => true, 'show_in_rest' => true,
        'sanitize_callback' => 'sanitize_text_field',
        'auth_callback' => $auth_cb,
    ) );
}

// Sync post meta (Gutenberg) to custom table after REST saves
add_action( 'rest_after_insert_post', 'gooloo_rbe_rest_sync_to_table', 10, 3 );
function gooloo_rbe_rest_sync_to_table( $post, $request, $creating ) {
    if ( 'post' !== $post->post_type ) return;
    $post_id = $post->ID;
    $value = get_post_meta( $post_id, '_gooloo_reviewbar_value', true );
    $price = get_post_meta( $post_id, '_gooloo_reviewbar_price', true );
    $brand = get_post_meta( $post_id, '_gooloo_reviewbar_brand', true );
    $sku   = get_post_meta( $post_id, '_gooloo_reviewbar_sku', true );
    if ( $value === '' && $price === '' && $brand === '' && $sku === '' ) return;
    global $wpdb; $table_name = $wpdb->prefix . 'review_bar_data';
    // Update or insert
    $exists = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(1) FROM $table_name WHERE post_id = %d", $post_id ) );
    if ( $exists ) {
        $wpdb->update( $table_name, array('value'=>$value,'price'=>$price,'brand'=>$brand,'sku'=>$sku), array('post_id'=>$post_id), array('%f','%f','%s','%s'), array('%d') );
    } else {
        $wpdb->insert( $table_name, array('post_id'=>$post_id,'value'=>$value,'price'=>$price,'brand'=>$brand,'sku'=>$sku), array('%d','%f','%f','%s','%s') );
    }
}

// Admin settings page (unchanged API)
add_action('admin_menu', 'review_bar_add_admin_menu');
function review_bar_add_admin_menu() {
    add_options_page( 'Review Bar Settings', 'Review Bar', 'manage_options', 'review-bar-settings', 'review_bar_settings_page' );
}
add_action('admin_init', 'review_bar_settings_init');
function review_bar_settings_init() {
    register_setting('review_bar_settings_group', 'review_bar_settings', 'review_bar_sanitize_settings');
    add_settings_section('review_bar_appearance_section', __('Appearance Settings','review-bar'), 'review_bar_appearance_section_callback', 'review-bar-settings');
    add_settings_section('review_bar_text_section', __('Text Settings','review-bar'), 'review_bar_text_section_callback', 'review-bar-settings');
    add_settings_field('line_color', __('Line Color','review-bar'), 'review_bar_line_color_callback', 'review-bar-settings', 'review_bar_appearance_section');
    add_settings_field('price_text', __('Price Text','review-bar'), 'review_bar_price_text_callback', 'review-bar-settings', 'review_bar_text_section');
    add_settings_field('rating_text', __('Rating Text','review-bar'), 'review_bar_rating_text_callback', 'review-bar-settings', 'review_bar_text_section');
    add_settings_field('euro_text', __('Currency Text','review-bar'), 'review_bar_euro_text_callback', 'review-bar-settings', 'review_bar_text_section');
    add_settings_field('additional_html', __('Additional HTML','review-bar'), 'review_bar_additional_html_callback', 'review-bar-settings', 'review_bar_text_section');
    add_settings_field('divider_style', __('Divider Style CSS','review-bar'), 'review_bar_divider_style_callback', 'review-bar-settings', 'review_bar_appearance_section');
}
function review_bar_sanitize_settings( $input ) {
    $sanitized = array();
    if ( isset( $input['line_color'] ) )   $sanitized['line_color'] = sanitize_hex_color( $input['line_color'] );
    if ( isset( $input['price_text'] ) )   $sanitized['price_text'] = wp_kses( $input['price_text'], array(), array() );
    if ( isset( $input['rating_text'] ) )  $sanitized['rating_text'] = wp_kses( $input['rating_text'], array(), array() );
    if ( isset( $input['euro_text'] ) )    $sanitized['euro_text'] = wp_kses( $input['euro_text'], array(), array() );
    if ( isset( $input['additional_html'] ) ) $sanitized['additional_html'] = wp_kses_post( $input['additional_html'] );
    if ( isset( $input['divider_style'] ) ) $sanitized['divider_style'] = sanitize_textarea_field( $input['divider_style'] );
    return $sanitized;
}
function review_bar_appearance_section_callback() { echo '<p>' . esc_html__( 'Customize the visual appearance of the review bar.', 'review-bar' ) . '</p>'; }
function review_bar_text_section_callback() { echo '<p>' . esc_html__( 'Customize the text displayed in the review bar.', 'review-bar' ) . '</p>'; }
function review_bar_line_color_callback() { $o = get_option('review_bar_settings'); $c = isset($o['line_color']) ? $o['line_color'] : '#ff0080'; echo '<input type="text" name="review_bar_settings[line_color]" value="' . esc_attr($c) . '" class="review-bar-color-picker" data-default-color="#ff0080" />'; }
function review_bar_price_text_callback() { $o = get_option('review_bar_settings'); $t = isset($o['price_text']) ? $o['price_text'] : 'Preis¹: '; echo '<input type="text" name="review_bar_settings[price_text]" value="' . esc_attr($t) . '" class="regular-text" />'; echo '<p class="description">' . esc_html__( 'Include trailing space (e.g. "Preis¹: " - note the space after the colon)', 'review-bar' ) . '</p>'; }
function review_bar_rating_text_callback() { $o = get_option('review_bar_settings'); $t = isset($o['rating_text']) ? $o['rating_text'] : 'Gesamtbewertung: '; echo '<input type="text" name="review_bar_settings[rating_text]" value="' . esc_attr($t) . '" class="regular-text" />'; echo '<p class="description">' . esc_html__( 'Include trailing space (e.g. "Gesamtbewertung: " - note the space after the colon)', 'review-bar' ) . '</p>'; }
function review_bar_euro_text_callback() { $o = get_option('review_bar_settings'); $t = isset($o['euro_text']) ? $o['euro_text'] : ' Euro'; echo '<input type="text" name="review_bar_settings[euro_text]" value="' . esc_attr($t) . '" class="regular-text" />'; echo '<p class="description">' . esc_html__( 'Include leading space (e.g. " Euro" - note the space before Euro)', 'review-bar' ) . '</p>'; }
function review_bar_additional_html_callback() { $o = get_option('review_bar_settings'); $h = isset($o['additional_html']) ? $o['additional_html'] : '<b>Du erhälst Informationen über diese Anzeige, indem Du <a href="/colorbanner">hier klickst</a>.</b>'; echo '<textarea name="review_bar_settings[additional_html]" rows="4" cols="50" class="large-text">' . esc_textarea($h) . '</textarea>'; echo '<p class="description">' . esc_html__( 'Additional HTML content displayed below the review bar. HTML tags are allowed.', 'review-bar' ) . '</p>'; }
function review_bar_divider_style_callback() { $o = get_option('review_bar_settings'); $s = isset($o['divider_style']) ? $o['divider_style'] : 'border-top: 2px solid {color}; margin: 10px 0;'; echo '<textarea name="review_bar_settings[divider_style]" rows="3" cols="50" class="large-text">' . esc_textarea($s) . '</textarea>'; echo '<p class="description">' . esc_html__( 'CSS style for the divider line. Use {color} placeholder for the selected color.', 'review-bar' ) . '</p>'; }
function review_bar_settings_page() { echo '<div class="wrap"><h1>' . esc_html( get_admin_page_title() ) . '</h1><form action="options.php" method="post">'; settings_fields('review_bar_settings_group'); do_settings_sections('review-bar-settings'); submit_button(); echo '</form></div>'; }

// Color picker only on settings page
add_action('admin_enqueue_scripts', 'review_bar_enqueue_color_picker');
function review_bar_enqueue_color_picker($hook) {
    if ( 'settings_page_review-bar-settings' !== $hook ) return;
    wp_enqueue_style('wp-color-picker');
    wp_enqueue_script('wp-color-picker');
    wp_add_inline_script('wp-color-picker', 'jQuery(function($){ $(".review-bar-color-picker").wpColorPicker(); });');
}

// Helper: get latest data for a post
function gooloo_rbe_get_data( $post_id ) {
    global $wpdb; $table = $wpdb->prefix . 'review_bar_data';
    $row = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table WHERE post_id = %d", $post_id ) );
    if ( ! $row ) return array( 'value'=>null, 'price'=>null, 'brand'=>'', 'sku'=>'' );
    return array( 'value'=>$row->value, 'price'=>$row->price, 'brand'=>$row->brand, 'sku'=>$row->sku );
}

// [reviewbar] shortcode (backward compatible)
function reviewbar_shortcode( $atts ) {
    $atts = shortcode_atts( array( 'value'=>'', 'price'=>'', 'post_id'=>'' ), $atts, 'reviewbar' );
    if ( ( $atts['value'] === '' || $atts['price'] === '' ) ) {
        $pid = $atts['post_id'] !== '' ? (int) $atts['post_id'] : ( is_singular() ? get_the_ID() : 0 );
        if ( $pid ) {
            $d = gooloo_rbe_get_data( $pid );
            if ( $atts['value'] === '' && $d['value'] !== null ) $atts['value'] = (string) $d['value'];
            if ( $atts['price'] === '' && $d['price'] !== null ) $atts['price'] = (string) $d['price'];
        }
    }
    if ( empty($atts['value']) || empty($atts['price']) || (float)$atts['value'] == 0 || (float)$atts['price'] == 0 ) return '';

    $options = get_option('review_bar_settings');
    $line_color = isset($options['line_color']) ? $options['line_color'] : '#ff0080';
    $price_text = isset($options['price_text']) ? $options['price_text'] : 'Preis¹: ';
    $rating_text= isset($options['rating_text']) ? $options['rating_text'] : 'Gesamtbewertung: ';
    $euro_text  = isset($options['euro_text']) ? $options['euro_text'] : ' Euro';
    $additional_html = isset($options['additional_html']) ? $options['additional_html'] : '';
    $divider_style = isset($options['divider_style']) ? str_replace('{color}', $line_color, $options['divider_style']) : 'border-top: 2px solid '.$line_color.'; margin: 10px 0;';

    $val = (float) $atts['value'];
    $color = ( $val <= 30 ? 'red' : ( $val <= 60 ? 'orange' : ( $val <= 75 ? 'yellow' : 'green' ) ) );
    $value_text = $rating_text . esc_html( $val ) . ' %';
    $price_text_formatted = number_format( (float) $atts['price'], 2, ',', '.' ) . $euro_text;

    $out  = '<div class="gooloo-reviewbar-container">';
    $out .= '<div class="gooloo-reviewbar-divider" style="' . esc_attr( $divider_style ) . '"></div>';
    $out .= '<span class="gooloo-reviewbar-text">' . esc_html( $price_text ) . esc_html( $price_text_formatted ) . ' | ' . esc_html( $value_text ) . '</span>';
    $out .= '<div class="gooloo-reviewbar" role="img" aria-label="' . esc_attr( $value_text ) . '" style="background-color:' . esc_attr( $color ) . '; width:' . esc_attr( $val ) . '%; height:20px;"></div>';
    $out .= '</div>';
    if ( $additional_html ) $out .= $additional_html;
    return $out;
}
add_shortcode( 'reviewbar', 'reviewbar_shortcode' );

// [toplist] shortcode (refined, safer output)
function toplist_shortcode( $atts ) {
    global $wpdb; $table = $wpdb->prefix . 'review_bar_data';
    $atts = shortcode_atts( array( 'rating'=>'', 'price'=>'', 'limit'=>10 ), $atts, 'toplist' );
    $where = array( 'p.post_status = "publish"' );
    $vals = array();
    if ( ! empty( $atts['rating'] ) ) {
        $r = array_map( 'floatval', explode('-', $atts['rating']) );
        if ( count($r) === 2 ) { $where[] = 'r.value BETWEEN %f AND %f'; $vals[] = $r[0]; $vals[] = $r[1]; }
    }
    if ( ! empty( $atts['price'] ) ) { $where[] = 'r.price <= %f'; $vals[] = (float) $atts['price']; }
    $where_sql = 'WHERE ' . implode( ' AND ', $where );
    $sql = $wpdb->prepare(
        "SELECT r.*, p.post_date, p.post_title FROM $table r JOIN {$wpdb->posts} p ON r.post_id=p.ID $where_sql ORDER BY p.post_date DESC LIMIT %d",
        array_merge( $vals, array( (int) $atts['limit'] ) )
    );
    $rows = $wpdb->get_results( $sql );
    if ( ! $rows ) return '';

    $opts = get_option('review_bar_settings');
    $line_color = isset($opts['line_color']) ? $opts['line_color'] : '#ff0080';

    static $inst = 0; $inst++;
    $wrap_id = 'gooloo-toplist-' . $inst;

    wp_enqueue_style( 'gooloo-toplist-css', GOOLOO_RBE_URL . 'assets/css/toplist.css', array(), GOOLOO_RBE_VERSION );
    wp_enqueue_script( 'gooloo-toplist-js', GOOLOO_RBE_URL . 'assets/js/toplist.js', array(), GOOLOO_RBE_VERSION, true );

    ob_start();
    ?>
    <div id="<?php echo esc_attr($wrap_id); ?>" class="gooloo-toplist-container" data-items-per-view="3">
        <div class="gooloo-toplist-slider">
            <?php foreach ( $rows as $item ): $color = ( $item->value <= 30 ? 'red' : ( $item->value <= 60 ? 'orange' : ( $item->value <= 75 ? 'yellow' : 'green' ) ) ); ?>
                <div class="gooloo-toplist-item">
                    <div class="gooloo-toplist-thumb"><?php echo get_the_post_thumbnail( $item->post_id, 'medium' ); ?></div>
                    <div class="gooloo-toplist-info">
                        <span class="gooloo-toplist-rating" style="color:<?php echo esc_attr($color); ?>;"><?php echo esc_html( number_format( (float)$item->value, 0 ) ); ?> %</span>
                        <span class="gooloo-toplist-price"><?php echo esc_html( number_format( (float)$item->price, 2, ',', '.' ) ); ?> &euro;</span>
                    </div>
                    <div class="gooloo-toplist-title"><?php echo esc_html( get_the_title( $item->post_id ) ); ?></div>
                    <div class="gooloo-toplist-excerpt"><?php echo esc_html( wp_trim_words( wp_strip_all_tags( get_post_field( 'post_content', $item->post_id ) ), 30 ) ); ?></div>
                    <div class="gooloo-toplist-button-wrapper"><a href="<?php echo esc_url( get_permalink( $item->post_id ) ); ?>" class="gooloo-toplist-readmore" style="background-color:<?php echo esc_attr($line_color); ?>;">Ansehen</a></div>
                </div>
            <?php endforeach; ?>
        </div>
        <div class="gooloo-toplist-nav">
            <button type="button" class="gooloo-toplist-prev" aria-label="Vorherige">&#x2039;</button>
            <button type="button" class="gooloo-toplist-next" aria-label="Nächste">&#x203A;</button>
        </div>
    </div>
    <script>window.GOOLOO_TOPLIST = (window.GOOLOO_TOPLIST||[]); window.GOOLOO_TOPLIST.push('<?php echo esc_js($wrap_id); ?>');</script>
    <?php
    return (string) ob_get_clean();
}
add_shortcode( 'toplist', 'toplist_shortcode' );

// Classic Editor meta boxes
add_action('add_meta_boxes', 'add_reviewbar_meta_boxes');
function add_reviewbar_meta_boxes() {
    add_meta_box('reviewbar_meta_box', 'Review Bar', 'reviewbar_meta_box_callback', 'post', 'advanced', 'high');
    add_meta_box('brand_meta_box', 'Brand', 'brand_meta_box_callback', 'post', 'advanced', 'high');
    add_meta_box('sku_meta_box', 'SKU/EAN/ISBN', 'sku_meta_box_callback', 'post', 'advanced', 'high');
}
function reviewbar_meta_box_callback($post) {
    global $wpdb; $table = $wpdb->prefix . 'review_bar_data';
    $data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table WHERE post_id = %d", $post->ID ) );
    $value = $data && $data->value !== null ? $data->value : '';
    $price = $data && $data->price !== null ? $data->price : '';
    wp_nonce_field('reviewbar_meta_box_nonce', 'reviewbar_meta_box_nonce');
    echo '<table class="form-table">'
       . '<tr><th scope="row"><label for="reviewbar_value">Value (%):</label></th><td><input type="number" step="0.01" min="0" max="100" id="reviewbar_value" name="reviewbar_value" value="' . esc_attr($value) . '" class="regular-text" /></td></tr>'
       . '<tr><th scope="row"><label for="reviewbar_price">Price:</label></th><td><input type="number" step="0.01" min="0" id="reviewbar_price" name="reviewbar_price" value="' . esc_attr($price) . '" class="regular-text" /></td></tr>'
       . '</table>';
}
function brand_meta_box_callback($post) {
    global $wpdb; $table = $wpdb->prefix . 'review_bar_data';
    $data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table WHERE post_id = %d", $post->ID ) );
    $brand = $data ? $data->brand : '';
    wp_nonce_field('brand_meta_box_nonce', 'brand_meta_box_nonce');
    echo '<table class="form-table"><tr><th scope="row"><label for="brand">Brand:</label></th><td><input type="text" id="brand" name="brand" value="' . esc_attr($brand) . '" class="regular-text" /></td></tr></table>';
}
function sku_meta_box_callback($post) {
    global $wpdb; $table = $wpdb->prefix . 'review_bar_data';
    $data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table WHERE post_id = %d", $post->ID ) );
    $sku = $data ? $data->sku : '';
    echo '<table class="form-table"><tr><th scope="row"><label for="sku">SKU/EAN/ISBN:</label></th><td><input type="text" id="sku" name="sku" value="' . esc_attr($sku) . '" class="regular-text" /></td></tr></table>';
}

// Robust save handler for Classic Editor
add_action('save_post_post', 'save_reviewbar_meta_box', 20, 2);
add_action('save_post', 'save_reviewbar_meta_box', 20, 2);
function save_reviewbar_meta_box($post_id, $post = null) {
    if ( ! isset($_POST['reviewbar_meta_box_nonce']) || ! wp_verify_nonce($_POST['reviewbar_meta_box_nonce'], 'reviewbar_meta_box_nonce') ) return;
    if ( function_exists('wp_is_post_autosave') && wp_is_post_autosave($post_id) ) return;
    if ( function_exists('wp_is_post_revision') && wp_is_post_revision($post_id) ) return;
    if ( ! isset($_POST['post_type']) || $_POST['post_type'] !== 'post' ) return;
    if ( ! current_user_can('edit_post', $post_id) ) return;

    global $wpdb; $table = $wpdb->prefix . 'review_bar_data';
    $value = isset($_POST['reviewbar_value']) && $_POST['reviewbar_value'] !== '' ? max(0, min(100, (float) str_replace(',', '.', $_POST['reviewbar_value']) )) : null;
    $price = isset($_POST['reviewbar_price']) && $_POST['reviewbar_price'] !== '' ? max(0, (float) str_replace(',', '.', $_POST['reviewbar_price']) ) : null;
    $brand = isset($_POST['brand']) ? sanitize_text_field( $_POST['brand'] ) : '';
    $sku   = isset($_POST['sku']) ? sanitize_text_field( $_POST['sku'] ) : '';

    $exists = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(1) FROM $table WHERE post_id = %d", $post_id ) );
    if ( $exists ) {
        $wpdb->update( $table, array('value'=>$value,'price'=>$price,'brand'=>$brand,'sku'=>$sku), array('post_id'=>$post_id), array('%f','%f','%s','%s'), array('%d') );
    } else {
        $wpdb->insert( $table, array('post_id'=>$post_id,'value'=>$value,'price'=>$price,'brand'=>$brand,'sku'=>$sku), array('%d','%f','%f','%s','%s') );
    }
    if ( $wpdb->last_error ) error_log('Review Bar Plugin Error: ' . $wpdb->last_error);

    if ( $value !== null ) update_post_meta( $post_id, '_gooloo_reviewbar_value', $value );
    if ( $price !== null ) update_post_meta( $post_id, '_gooloo_reviewbar_price', $price );
    update_post_meta( $post_id, '_gooloo_reviewbar_brand', $brand );
    update_post_meta( $post_id, '_gooloo_reviewbar_sku', $sku );

    if ( $value !== null ) update_post_meta( $post_id, 'reviewbar_value', $value );
    if ( $price !== null ) update_post_meta( $post_id, 'reviewbar_price', $price );
}

// Auto-append review bar after content (unchanged behavior)
add_filter('the_content', 'add_reviewbar_shortcode');
function add_reviewbar_shortcode($content) {
    if ( ! is_singular('post') ) return $content;
    global $wpdb, $post; $table = $wpdb->prefix . 'review_bar_data';
    $data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table WHERE post_id = %d", $post->ID ) );
    if ( $data && $data->value !== null && $data->price !== null && $data->value != 0 && $data->price != 0 ) {
        $content .= do_shortcode( '[reviewbar value="' . esc_attr($data->value) . '" price="' . esc_attr($data->price) . '"]' );
    }
    return $content;
}

// Front-end minimal styles
add_action('wp_head', 'add_reviewbar_styles');
function add_reviewbar_styles() {
    $options = get_option('review_bar_settings');
    $line_color = isset($options['line_color']) ? $options['line_color'] : '#ff0080';
    echo '<style>
        .gooloo-reviewbar-container { width: 100%; padding: 10px; box-sizing: border-box; }
        .gooloo-reviewbar-divider { border-top: 2px solid ' . esc_attr($line_color) . '; margin: 10px 0; }
        .gooloo-reviewbar-text { font-size: 24px; font-weight: bold; color: #000; margin-bottom: 10px; display: block; }
        .gooloo-reviewbar { height: 20px; border-radius: 10px; transition: width 0.5s ease-in-out; }
        @media (max-width: 768px) { .gooloo-reviewbar-container { padding: 5px; } .gooloo-reviewbar-text { font-size: 20px; } }
        @media (max-width: 480px) { .gooloo-reviewbar-container { padding: 5px; } .gooloo-reviewbar-text { font-size: 16px; } }
    </style>';
}

// Identifiers validation utilities
function validate_gtin($gtin) { if (empty($gtin)) return false; $gtin = preg_replace('/[^0-9]/','',$gtin); $len = strlen($gtin); if ( ! in_array($len, array(8,12,13,14), true) ) return false; $sum=0; for($i=0;$i<$len-1;$i++){ $digit=(int)$gtin[$i]; $position = ($len==8)? $i+1 : $i; $mult = ($position%2==0)?1:3; $sum += $digit*$mult; } $checksum = (10 - ($sum%10))%10; return $checksum === (int)$gtin[$len-1]; }
function validate_isbn($isbn) { if (empty($isbn)) return false; $isbn = preg_replace('/[-\s]/','',$isbn); if (strlen($isbn)==10) return validate_isbn10($isbn); if (strlen($isbn)==13) return validate_isbn13($isbn); return false; }
function validate_isbn10($isbn){ if(strlen($isbn)!=10) return false; $sum=0; for($i=0;$i<9;$i++){ if(!is_numeric($isbn[$i])) return false; $sum += (int)$isbn[$i]*(10-$i);} $checksum = $sum%11; $check=$isbn[9]; if($checksum==0) return $check=='0'; if($checksum==1) return strtoupper($check)=='X'; return (int)$check==(11-$checksum);} 
function validate_isbn13($isbn){ if(strlen($isbn)!=13 || !is_numeric($isbn)) return false; if(substr($isbn,0,3)!=='978' && substr($isbn,0,3)!=='979') return false; $sum=0; for($i=0;$i<12;$i++){ $sum += (int)$isbn[$i]*( $i%2==0 ? 1 : 3 ); } $checksum=(10-($sum%10))%10; return (int)$isbn[12]===$checksum; }

// Schema.org output
add_action('wp_head', 'add_product_review_schema', 20);
function add_product_review_schema() {
    if ( ! is_singular('post') ) return; global $wpdb, $post; $table = $wpdb->prefix . 'review_bar_data';
    $data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table WHERE post_id = %d", $post->ID ) );
    if ( ! $data || $data->value === null || $data->price === null || $data->value <= 0 || $data->price <= 0 ) return;
    $featured_image = get_the_post_thumbnail_url( $post->ID, 'full' ); if ( ! $featured_image ) $featured_image = get_site_url() . '/wp-content/uploads/default-product.jpg';
    $description = get_the_excerpt( $post->ID ); if ( ! $description ) $description = wp_trim_words( wp_strip_all_tags( get_the_content(null,false,$post) ), 30 );
    $author_name = get_the_author_meta( 'display_name', $post->post_author ); if ( ! $author_name ) $author_name = get_bloginfo( 'name' );

    $schema = array(
        '@context' => 'https://schema.org/',
        '@type'    => 'Product',
        'name'        => get_the_title( $post->ID ),
        'description' => $description,
        'image'       => array( $featured_image ),
        'url'         => get_permalink( $post->ID ),
    );
    if ( ! empty( $data->brand ) ) {
        $schema['brand'] = array( '@type' => 'Brand', 'name' => $data->brand );
    }
    if ( ! empty( $data->sku ) ) {
        $sku_value = trim( $data->sku );
        if ( validate_isbn( $sku_value ) ) { $schema['isbn'] = preg_replace('/[-\s]/','',$sku_value); }
        elseif ( validate_gtin( $sku_value ) ) { $clean = preg_replace('/[^0-9]/','',$sku_value); $schema['gtin'.strlen($clean)] = $clean; }
        else { $schema['sku'] = $sku_value; }
    }
    $schema['offers'] = array(
        '@type' => 'Offer',
        'price' => number_format( (float)$data->price, 2, '.', '' ),
        'priceCurrency' => 'EUR',
        'availability' => 'https://schema.org/InStock',
        'url' => get_permalink( $post->ID ),
    );
    $rating_value = (float) $data->value;
    $schema['review'] = array(
        '@type' => 'Review',
        'reviewRating' => array(
            '@type' => 'Rating',
            'ratingValue' => $rating_value,
            'bestRating' => 100,
            'worstRating' => 0,
        ),
        'author' => array( '@type' => 'Person', 'name' => $author_name ),
        'datePublished' => get_the_date( 'c', $post->ID ),
        'reviewBody' => wp_trim_words( $description, 20 ),
    );
    if ( $rating_value > 0 && $rating_value <= 100 ) {
        $schema['aggregateRating'] = array(
            '@type' => 'AggregateRating',
            'ratingValue' => $rating_value,
            'bestRating' => 100,
            'worstRating' => 0,
            'ratingCount' => 1,
        );
    }
    echo '<script type="application/ld+json" class="review-bar-schema">' . wp_json_encode( $schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT ) . '</script>' . "\n";
}

// Settings link in plugins list
add_filter( 'plugin_action_links_' . plugin_basename(__FILE__), 'review_bar_add_settings_link' );
function review_bar_add_settings_link( $links ) { $settings_link = '<a href="' . esc_url( admin_url('options-general.php?page=review-bar-settings') ) . '">' . esc_html__( 'Settings' ) . '</a>'; array_unshift( $links, $settings_link ); return $links; }

// Version option updater (backward-compatible)
add_action('admin_init', 'review_bar_check_version');
function review_bar_check_version() { $installed = get_option('review_bar_version','1.0'); if ( version_compare( $installed, GOOLOO_RBE_VERSION, '<' ) ) { update_option('review_bar_version', GOOLOO_RBE_VERSION); } }

// Editor assets: add sidebar panel to Block Editor to manage meta that syncs to DB
add_action( 'enqueue_block_editor_assets', 'gooloo_rbe_enqueue_editor_assets' );
function gooloo_rbe_enqueue_editor_assets() {
    wp_enqueue_script( 'gooloo-reviewbar-editor', GOOLOO_RBE_URL . 'assets/js/editor.js', array( 'wp-element','wp-components','wp-data','wp-edit-post','wp-plugins','wp-i18n' ), GOOLOO_RBE_VERSION, true );
}
