<?php
/*
Plugin Name: Auto Bluesky Poster
Description: Automatically posts to Bluesky when a WordPress post is published, with support for OG tags and image thumbnails.
Version: 2.0
*/

// Prevent direct access
if (!defined('ABSPATH')) exit;

// --- UTILITIES ---

/**
 * Custom logging function.
 * Logs messages to wp-content/bluesky-debug.log when WP_DEBUG is true.
 */
function bluesky_log($message, $level = 'info') {
    if (WP_DEBUG === true) {
        $log_file = WP_CONTENT_DIR . '/bluesky-debug.log';
        $timestamp = current_time('mysql');
        $formatted_message = sprintf("[%s] %s: %s\n",
            $timestamp,
            strtoupper($level),
            is_array($message) || is_object($message) ? print_r($message, true) : $message
        );
        error_log($formatted_message, 3, $log_file);
    }
}

// --- PLUGIN SETUP AND SETTINGS PAGE ---

// Register activation hook to set default options
register_activation_hook(__FILE__, function() {
    add_option('bluesky_handle', '');
    add_option('bluesky_app_password', '');
    add_option('bluesky_before_text', 'New on my blog:');
    add_option('bluesky_after_text', '#blog');
    add_option('bluesky_last_error', '');
    bluesky_log('Plugin activated');
});

// Add settings page to the admin menu
add_action('admin_menu', function() {
    add_options_page(
        'Bluesky Settings',
        'Bluesky Auto Post',
        'manage_options',
        'bluesky-settings',
        'bluesky_settings_page_html'
    );
});

// Register settings
add_action('admin_init', function() {
    register_setting('bluesky_settings', 'bluesky_handle', function($handle) {
        return trim(str_replace(['@', ' '], '', $handle));
    });
    register_setting('bluesky_settings', 'bluesky_app_password');
    register_setting('bluesky_settings', 'bluesky_before_text', 'sanitize_text_field');
    register_setting('bluesky_settings', 'bluesky_after_text', 'sanitize_text_field');
});

// HTML for the settings page
function bluesky_settings_page_html() {
    if (!current_user_can('manage_options')) return;

    if (isset($_POST['test_connection'])) {
        check_admin_referer('bluesky_test_connection');
        $test_result = test_bluesky_connection();
        echo $test_result
            ? '<div class="notice notice-success"><p>Connection successful!</p></div>'
            : '<div class="notice notice-error"><p>Connection failed: ' . esc_html(get_option('bluesky_last_error')) . '</p></div>';
    }
    ?>
    <div class="wrap">
        <h2>Bluesky Auto Post Settings</h2>
        <form method="post" action="options.php">
            <?php settings_fields('bluesky_settings'); ?>
            <table class="form-table">
                <tr>
                    <th><label for="bluesky_handle">Bluesky Handle</label></th>
                    <td>
                        <input type="text" id="bluesky_handle" name="bluesky_handle" value="<?php echo esc_attr(get_option('bluesky_handle')); ?>" class="regular-text" />
                        <p class="description">Enter your full handle (e.g., username.bsky.social).</p>
                    </td>
                </tr>
                <tr>
                    <th><label for="bluesky_app_password">App Password</label></th>
                    <td>
                        <input type="password" id="bluesky_app_password" name="bluesky_app_password" value="<?php echo esc_attr(get_option('bluesky_app_password')); ?>" class="regular-text" />
                        <p class="description">Create an app-specific password from your Bluesky settings.</p>
                    </td>
                </tr>
                <tr>
                    <th><label for="bluesky_before_text">Text Before Post Title</label></th>
                    <td>
                        <input type="text" id="bluesky_before_text" name="bluesky_before_text" value="<?php echo esc_attr(get_option('bluesky_before_text')); ?>" class="regular-text" />
                        <p class="description">Text that appears before the post title (can include #hashtags).</p>
                    </td>
                </tr>
                <tr>
                    <th><label for="bluesky_after_text">Text After URL</label></th>
                    <td>
                        <input type="text" id="bluesky_after_text" name="bluesky_after_text" value="<?php echo esc_attr(get_option('bluesky_after_text')); ?>" class="regular-text" />
                        <p class="description">Text that appears after the URL (can include #hashtags).</p>
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
        <hr/>
        <form method="post">
            <?php wp_nonce_field('bluesky_test_connection'); ?>
            <input type="hidden" name="test_connection" value="1">
            <?php submit_button('Test Connection', 'secondary'); ?>
        </form>
    </div>
    <?php
}

// --- BLUESKY API COMMUNICATION ---

/**
 * Creates a new session with the Bluesky API.
 * @return array Session data including 'did' and 'accessJwt'.
 * @throws Exception on failure.
 */
function create_bluesky_session() {
    $handle = get_option('bluesky_handle');
    $password = get_option('bluesky_app_password');
    if (empty($handle) || empty($password)) {
        throw new Exception('Bluesky handle or app password is not set.');
    }

    $response = wp_remote_post('https://bsky.social/xrpc/com.atproto.server.createSession', [
        'headers' => ['Content-Type' => 'application/json'],
        'body' => json_encode(['identifier' => $handle, 'password' => $password]),
        'timeout' => 30
    ]);

    if (is_wp_error($response)) {
        throw new Exception('Session API request failed: ' . $response->get_error_message());
    }

    $body = json_decode(wp_remote_retrieve_body($response), true);
    if (!isset($body['did'], $body['accessJwt'])) {
        throw new Exception('Invalid session response from Bluesky: ' . print_r($body, true));
    }
    return $body;
}

/**
 * Fetches the post URL and parses its OG meta tags.
 * @param string $url The URL of the post.
 * @return array An associative array of OG tags.
 */
function fetch_and_parse_og_tags($url) {
    bluesky_log("Fetching OG tags from: $url");
    $response = wp_remote_get($url, ['timeout' => 20]);
    if (is_wp_error($response)) {
        bluesky_log("Failed to fetch post content: " . $response->get_error_message(), 'warn');
        return [];
    }

    $html = wp_remote_retrieve_body($response);
    if (!class_exists('DOMDocument')) {
        bluesky_log("DOMDocument class not found. Cannot parse OG tags.", 'error');
        return [];
    }

    $doc = new DOMDocument();
    @$doc->loadHTML('<?xml encoding="utf-8" ?>' . $html);
    $xpath = new DOMXPath($doc);
    $query = '//*/meta[starts-with(@property, \'og:\')]';
    $metas = $xpath->query($query);
    $og_tags = [];
    foreach ($metas as $meta) {
        $property = $meta->getAttribute('property');
        $content = $meta->getAttribute('content');
        $og_tags[$property] = $content;
    }
    bluesky_log("Found OG tags: " . print_r($og_tags, true));
    return $og_tags;
}

/**
 * Uploads an image to the Bluesky blob server.
 * @param string $image_url URL of the image to upload.
 * @param string $access_jwt The API access token.
 * @return array|null The blob reference object or null on failure.
 */
function upload_image_to_bluesky($image_url, $access_jwt) {
    bluesky_log("Uploading image from: $image_url");
    $image_response = wp_remote_get($image_url, ['timeout' => 30]);
    if (is_wp_error($image_response)) {
        bluesky_log("Failed to download image: " . $image_response->get_error_message(), 'warn');
        return null;
    }

    $image_data = wp_remote_retrieve_body($image_response);
    $mime_type = wp_remote_retrieve_header($image_response, 'content-type');
    if (empty($mime_type) || strpos($mime_type, 'image/') !== 0) {
        $mime_type = 'image/jpeg'; // Fallback MIME type
        bluesky_log("Could not determine image MIME type, falling back to $mime_type.", 'warn');
    }

    $upload_response = wp_remote_post('https://bsky.social/xrpc/com.atproto.repo.uploadBlob', [
        'headers' => [
            'Authorization' => 'Bearer ' . $access_jwt,
            'Content-Type' => $mime_type
        ],
        'body' => $image_data,
        'timeout' => 45
    ]);

    if (is_wp_error($upload_response)) {
        bluesky_log("Image upload failed: " . $upload_response->get_error_message(), 'error');
        return null;
    }

    $body = json_decode(wp_remote_retrieve_body($upload_response), true);
    if (isset($body['blob'])) {
        bluesky_log("Image uploaded successfully.");
        return $body['blob'];
    }
    
    bluesky_log("Invalid blob response: " . print_r($body, true), 'error');
    return null;
}

/**
 * Creates rich text facets for links and hashtags.
 * @param string $text The full text of the post.
 * @param string $url The URL to be linked.
 * @return array The list of facet objects.
 */
function create_facets($text, $url) {
    $facets = [];
    $text_bytes = mb_convert_encoding($text, 'UTF-8');

    // Link facet
    $url_start = strpos($text_bytes, $url);
    if ($url_start !== false) {
        $facets[] = [
            'index' => ['byteStart' => $url_start, 'byteEnd' => $url_start + strlen($url)],
            'features' => [['$type' => 'app.bsky.richtext.facet#link', 'uri' => $url]]
        ];
    }

    // Hashtag facets using precise byte offsets
    if (preg_match_all('/#[\w\p{L}\p{N}_-]+/u', $text, $matches, PREG_OFFSET_CAPTURE)) {
        foreach ($matches[0] as $match) {
            $tag_text = $match[0];
            $tag_start = mb_strlen(substr($text, 0, $match[1]), 'UTF-8');
            
            // Convert character offset to byte offset for the API
            $byte_start = strlen(mb_convert_encoding(substr($text, 0, $match[1]), 'UTF-8'));
            $byte_end = $byte_start + strlen(mb_convert_encoding($tag_text, 'UTF-8'));

            $facets[] = [
                'index' => ['byteStart' => $byte_start, 'byteEnd' => $byte_end],
                'features' => [['$type' => 'app.bsky.richtext.facet#tag', 'tag' => substr($tag_text, 1)]]
            ];
        }
    }
    return $facets;
}

// --- CORE POSTING LOGIC ---

/**
 * Main function to construct and send the post to Bluesky.
 * @param WP_Post $post The WordPress post object.
 * @return bool True on success, false on failure.
 */
function post_to_bluesky($post) {
    try {
        bluesky_log('Starting Bluesky post process for: ' . $post->post_title);
        if (get_post_meta($post->ID, '_posted_to_bluesky', true)) {
            bluesky_log('Post has already been shared. Skipping.');
            return false;
        }

        $session = create_bluesky_session();
        $url = get_permalink($post->ID);
        
        // Fetch OG data
        $og_tags = fetch_and_parse_og_tags($url);
        $card_title = !empty($og_tags['og:title']) ? $og_tags['og:title'] : $post->post_title;
        $card_description = !empty($og_tags['og:description']) ? $og_tags['og:description'] : wp_trim_words(strip_tags($post->post_content), 50);

        // Build the text part of the post
        $before_text = get_option('bluesky_before_text', '');
        $after_text = get_option('bluesky_after_text', '');
        $text = trim(implode(' ', array_filter([$before_text, $post->post_title, $url, $after_text])));

        // Prepare the link card embed
        $embed = [
            '$type' => 'app.bsky.embed.external',
            'external' => [
                'uri' => $url,
                'title' => $card_title,
                'description' => $card_description,
            ]
        ];

        // If an OG image exists, upload it and add it to the embed card
        if (!empty($og_tags['og:image'])) {
            $image_blob = upload_image_to_bluesky($og_tags['og:image'], $session['accessJwt']);
            if ($image_blob) {
                $embed['external']['thumb'] = $image_blob;
            }
        }

        // Create the final post record
        $post_data = [
            'repo' => $session['did'],
            'collection' => 'app.bsky.feed.post',
            'record' => [
                '$type' => 'app.bsky.feed.post',
                'text' => $text,
                'createdAt' => gmdate('Y-m-d\TH:i:s.v\Z'),
                'facets' => create_facets($text, $url),
                'embed' => $embed,
                'langs' => [substr(get_locale(), 0, 2)] // e.g., 'en' from 'en_US'
            ]
        ];

        $response = wp_remote_post('https://bsky.social/xrpc/com.atproto.repo.createRecord', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $session['accessJwt']
            ],
            'body' => json_encode($post_data),
            'timeout' => 45
        ]);

        if (is_wp_error($response)) {
            throw new Exception('Post creation failed: ' . $response->get_error_message());
        }

        $result = json_decode(wp_remote_retrieve_body($response), true);
        if (wp_remote_retrieve_response_code($response) >= 400) {
            throw new Exception('Bluesky API Error: ' . ($result['message'] ?? print_r($result, true)));
        }

        if (isset($result['uri'])) {
            add_post_meta($post->ID, '_posted_to_bluesky', true, true);
            bluesky_log('Successfully posted to Bluesky. URI: ' . $result['uri']);
            return true;
        }

        throw new Exception('Invalid post response from Bluesky: ' . print_r($result, true));

    } catch (Exception $e) {
        bluesky_log('Error posting to Bluesky: ' . $e->getMessage(), 'error');
        update_option('bluesky_last_error', $e->getMessage());
        return false;
    }
}

/**
 * Tests the connection to Bluesky by creating a session.
 * @return bool True if connection is successful.
 */
function test_bluesky_connection() {
    try {
        create_bluesky_session();
        update_option('bluesky_last_error', '');
        return true;
    } catch (Exception $e) {
        update_option('bluesky_last_error', $e->getMessage());
        return false;
    }
}

// --- WORDPRESS ACTION HOOKS ---

/**
 * Main hook handler for when a post is published.
 * @param int $post_id The ID of the post.
 * @param WP_Post $post The post object.
 */
function on_post_published($post_id, $post) {
    // Basic checks to avoid running on revisions, autosaves, or non-posts
    if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id) || $post->post_type !== 'post') {
        return;
    }

    // Only run on initial publication, not updates
    if (get_post_meta($post_id, '_posted_to_bluesky', true)) {
        return;
    }
    
    // Check if the post status is 'publish'
    if ($post->post_status === 'publish') {
        post_to_bluesky($post);
    }
}
add_action('publish_post', 'on_post_published', 10, 2);

/**
 * Hook for posts that were scheduled and are now being published.
 * @param int $post_id The ID of the post.
 */
function on_future_post_published($post_id) {
    $post = get_post($post_id);
    on_post_published($post_id, $post);
}
add_action('publish_future_post', 'on_future_post_published', 10, 1);
