class Category_Link_Optimizer {
    private static $instance = null;
    private $category_links = [];
    private $cache_key = 'category_links_cache';
    private $cache_time = 3600; // 1 hour
    private $excluded_words = ['Amazon', 'gooloo'];
    private $max_links_per_category = 2;

    private function __construct() {
        add_filter('the_content', [$this, 'link_categories_in_content'], 99);
        add_action('edited_category', [$this, 'clear_cache']);
        add_action('create_category', [$this, 'clear_cache']);
        add_action('delete_category', [$this, 'clear_cache']);
    }

    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function get_cached_category_links() {
        if (!empty($this->category_links)) {
            return $this->category_links;
        }

        $this->category_links = wp_cache_get($this->cache_key);

        if (false === $this->category_links) {
            $this->category_links = $this->generate_category_links();
            wp_cache_set($this->cache_key, $this->category_links, '', $this->cache_time);
        }

        return $this->category_links;
    }

    private function generate_category_links() {
        $categories = get_categories(['hide_empty' => false, 'orderby' => 'name', 'order' => 'ASC']);
        $links = [];

        foreach ($categories as $category) {
            if (!in_array($category->name, $this->excluded_words)) {
                $links[$category->name] = get_category_link($category->term_id);
            }
        }

        return $links;
    }

    private function replace_category_with_limit($content, $name, $url) {
        $count = 0;
        $pattern = '/\b' . preg_quote($name, '/') . '\b(?![^<]*>)/i';
        
        return preg_replace_callback($pattern, function($matches) use (&$count, $url, $name) {
            if ($count < $this->max_links_per_category) {
                $count++;
                return '<a href="' . esc_url($url) . '">' . $name . '</a>';
            }
            return $matches[0];
        }, $content);
    }

    public function link_categories_in_content($content) {
        // Check if it's a single post and specifically of post type 'post'
        if (!is_single() || get_post_type() !== 'post') {
            return $content;
        }

        $category_links = $this->get_cached_category_links();

        // Sort category names by length (descending) to match longer names first
        uksort($category_links, function($a, $b) {
            return strlen($b) - strlen($a);
        });

        // Split content by the amazon search div
        $parts = preg_split('/<div class=["\']amazon-search["\'].*?<\/div>/s', $content);
        
        // Process each part separately
        foreach ($parts as &$part) {
            foreach ($category_links as $name => $url) {
                $part = $this->replace_category_with_limit($part, $name, $url);
            }
        }

        // Reassemble the content with the original amazon search divs
        $amazon_search_divs = [];
        preg_match_all('/<div class=["\']amazon-search["\'].*?<\/div>/s', $content, $amazon_search_divs);
        
        $final_content = '';
        for ($i = 0; $i < count($parts); $i++) {
            $final_content .= $parts[$i];
            if (isset($amazon_search_divs[0][$i])) {
                $final_content .= $amazon_search_divs[0][$i];
            }
        }

        return $final_content;
    }

    public function clear_cache() {
        wp_cache_delete($this->cache_key);
        $this->category_links = [];

        // Clear LiteSpeed Cache if it's active
        if (class_exists('LiteSpeed_Cache_API')) {
            LiteSpeed_Cache_API::purge_all();
        }
    }
}

// Initialize the Category Link Optimizer
add_action('init', function() {
    Category_Link_Optimizer::get_instance();
});